import { shallow } from 'enzyme';
import NextLink from 'next/link';
import { pathnameFromRouteName } from '../pathnameFromRouteName';
import { RouteName } from '../routes';
import { TachyonLink } from '.';

// These tests use "shallow" from enzyme directly due to the complex types
describe(TachyonLink, () => {
  it('renders just an href for a static page path', () => {
    const wrapper = shallow(<TachyonLink route={RouteName.Homepage} />);

    const nextLink = wrapper.find(NextLink);
    expect(nextLink).toHaveProp({
      as: undefined,
      href: pathnameFromRouteName(RouteName.Homepage),
    });
  });

  it('renders an href and as for a static page path with query params', () => {
    const wrapper = shallow(
      <TachyonLink query={{ foo: 'bar' }} route={RouteName.Homepage} />,
    );

    const nextLink = wrapper.find(NextLink);
    expect(nextLink).toHaveProp({
      as: `${pathnameFromRouteName(RouteName.Homepage)}?foo=bar`,
      href: `${pathnameFromRouteName(RouteName.Homepage)}?foo=bar`,
    });
  });

  it('renders an href and as for a dynamic page path', () => {
    const wrapper = shallow(
      <TachyonLink
        route={RouteName.Channel}
        routeParams={{ login: 'therealderekt' }}
      />,
    );

    const nextLink = wrapper.find(NextLink);
    expect(nextLink).toHaveProp({
      as: '/therealderekt',
      href: pathnameFromRouteName(RouteName.Channel),
    });
  });

  it('renders an href and as for a dynamic page path with query params', () => {
    const wrapper = shallow(
      <TachyonLink
        query={{ foo: 'bar' }}
        route={RouteName.Channel}
        routeParams={{ login: 'therealderekt' }}
      />,
    );

    const nextLink = wrapper.find(NextLink);
    expect(nextLink).toHaveProp({
      as: '/therealderekt?foo=bar',
      href: `${pathnameFromRouteName(RouteName.Channel)}?foo=bar`,
    });
  });

  it('renders just an anchor tag for external links with allowed protocol', () => {
    const wrapper = shallow(
      <TachyonLink
        route={RouteName.External}
        routeParams={{ href: 'https://twitch.tv' }}
      />,
    );

    expect(wrapper.find(NextLink)).not.toExist();
    expect(wrapper.is('a')).toBe(true);
    expect(wrapper.find('a')).toHaveProp({
      href: 'https://twitch.tv',
    });
  });

  it('renders just empty anchor tag for external links with non-allowed protocol', () => {
    const wrapper = shallow(
      <TachyonLink
        route={RouteName.External}
        routeParams={{ href: 'javascript:alert("pwned")' }}
      />,
    );

    expect(wrapper.find(NextLink)).not.toExist();
    expect(wrapper.is('a')).toBe(true);
    expect(wrapper.find('a')).toHaveProp({ href: undefined });
  });
});
