import type { TachyonPageContext, TachyonResponse } from 'tachyon-next-types';
import { pathnameFromRouteName } from '../pathnameFromRouteName';
import { RouteName } from '../routes';
import { pathnameFromContext } from '.';

function mockTachyonPageContext(
  overrides?: Partial<TachyonPageContext>,
): TachyonPageContext {
  const defaultContext: TachyonPageContext = {
    AppTree: () => null,
    asPath: '/',
    language: 'en-US',
    pathname: '/',
    query: {},
    req: {} as any,
  };

  return { ...defaultContext, ...overrides };
}

function mockServerResponse(
  statusCode: number,
  statusMessage: string,
): TachyonResponse {
  // This type is massive, so we mock this down to the bare minimum
  return {
    statusCode,
    statusMessage,
  } as TachyonResponse;
}

describe(pathnameFromContext, () => {
  it('returns a not found pathname if there is no context', () => {
    expect(pathnameFromContext(undefined)).toEqual(
      pathnameFromRouteName(RouteName.NotFound),
    );
  });

  it('returns a valid route name if the ctx has no client side errors', () => {
    const ctx = mockTachyonPageContext();
    expect(pathnameFromContext(ctx)).toEqual(
      pathnameFromRouteName(RouteName.Homepage),
    );
  });

  it('returns a valid route name if the ctx has no server side errors', () => {
    const ctx = mockTachyonPageContext({ res: mockServerResponse(200, 'OK') });
    expect(pathnameFromContext(ctx)).toEqual(
      pathnameFromRouteName(RouteName.Homepage),
    );
  });

  describe('when a context with a 4XX range server response is provided', () => {
    it('returns an internal error route name for a 400', () => {
      const ctx = mockTachyonPageContext({
        res: mockServerResponse(400, 'Bad Request'),
      });
      expect(pathnameFromContext(ctx)).toEqual(
        pathnameFromRouteName(RouteName.NotFound),
      );
    });

    it('returns an internal error route name for a 404', () => {
      const ctx = mockTachyonPageContext({
        res: mockServerResponse(404, 'Not Found'),
      });
      expect(pathnameFromContext(ctx)).toEqual(
        pathnameFromRouteName(RouteName.NotFound),
      );
    });

    it('returns an internal error route name for a 499', () => {
      const ctx = mockTachyonPageContext({
        res: mockServerResponse(499, 'Crazy error'),
      });
      expect(pathnameFromContext(ctx)).toEqual(
        pathnameFromRouteName(RouteName.NotFound),
      );
    });
  });
});
