import { random } from 'faker';
import { Enum } from 'tachyon-utils';
import { RouteLinkPathnames, RouteName } from '../routes';
import { routeNameFromRawPath } from '.';

describe(routeNameFromRawPath, () => {
  it.each(Enum.entries(RouteLinkPathnames))(
    'correctly matches for %s',
    (routeName: string, pathname: string) => {
      // coalesce all the channel home dupes
      const expected = routeName.includes('placeholder')
        ? RouteName.ChannelHome
        : routeName;
      const rawPath = pathname.replace(/\[(\w+)\]/g, random.alphaNumeric(5));
      expect(routeNameFromRawPath(rawPath)).toEqual(expected);
    },
  );

  type NonUpsellTestCase = {
    path: string;
    route: RouteName;
  };

  // sanity-checking the above generated tests and also validating
  // query param handling
  it.each`
    path                                     | route
    ${'/tfue'}                               | ${RouteName.Channel}
    ${'/tfue/home'}                          | ${RouteName.ChannelHome}
    ${'/directory/all'}                      | ${RouteName.GameDirectoryAll}
    ${'/directory/game/Fortnite'}            | ${RouteName.GameDirectory}
    ${'/search?term=counter'}                | ${RouteName.Search}
    ${'/app-shell?destination=%2Fdirectory'} | ${RouteName.AppShell}
  `('matches $path to $route', ({ path, route }: NonUpsellTestCase) => {
    expect(routeNameFromRawPath(path)).toEqual(route);
  });

  type UpsellTestCase = {
    path: string;
    upsell: RouteName.Upsell | RouteName.Upsell2;
  };

  it.each`
    path                      | upsell
    ${'/a/b'}                 | ${RouteName.Upsell}
    ${'/a/b/c'}               | ${RouteName.Upsell2}
    ${'/directory/some'}      | ${RouteName.Upsell}
    ${'/directory/some/more'} | ${RouteName.Upsell2}
    ${'/clip/some/more'}      | ${RouteName.Upsell2}
    ${'/videos/some/more'}    | ${RouteName.Upsell2}
  `('matches $path to $upsell', ({ path, upsell }: UpsellTestCase) => {
    expect(routeNameFromRawPath(path)).toEqual(upsell);
  });

  type NotFoundTestCase = {
    path: string;
  };

  it.each`
    path
    ${'/a/b/c/d'}
    ${'/a/b/c/d/e'}
    ${'/a/b/c/d/e/f'}
    ${'/directory/some/more/stuff'}
    ${'/clip/some/more/stuff'}
    ${'/videos/some/more/stuff'}
  `('matches $path to not found', ({ path }: NotFoundTestCase) => {
    expect(routeNameFromRawPath(path)).toEqual(RouteName.NotFound);
  });
});
