import { Enum } from 'tachyon-utils';

/**
 * A list of all route names in the app. This list must manually be updated when a new
 * page is added to the application. The values map to the route's equivalent Spade "location".
 */
export enum RouteName {
  AppShell = 'app_shell',
  Channel = 'channel',
  ChannelAbout = 'channel_about',
  ChannelClips = 'channel_clips',
  ChannelCollections = 'channel_collections_placeholder',
  ChannelFollowers = 'channel_followers_placeholder',
  ChannelFollowing = 'channel_following_placeholder',
  ChannelHome = 'channel_home',
  ChannelSchedule = 'channel_schedule',
  ChannelStandalone = 'channel_standalone',
  ChannelVideos = 'channel_videos',
  Clip = 'clip',
  ClipLegacy = 'clips_viewing',
  External = 'external',
  GameDirectory = 'directory_game',
  GameDirectoryAll = 'directory_game_all',
  GamesDirectory = 'directory_main_game',
  HealthCheck = 'health_check',
  Homepage = 'homepage',
  NotFound = 'not_found',
  ReportWizard = 'report',
  Search = 'search',
  Upsell = 'mobile_upsell',
  Upsell2 = 'mobile_upsell2',
  Vod = 'vod',
}

/**
 * List of route names that correspond to loading/forwarding states and not destinations
 */
export const INTERSTITIAL_ROUTE_NAMES = [RouteName.AppShell];

type RouteMap = { [key in RouteName]: string };

/**
 * A mapping of Route names to their Next equivalent static "pathname". These
 * map directly to the folder structure of src/pages/*.
 */
export const RouteLinkPathnames: RouteMap = {
  // Tip: Seeing surprising redirect behavior?
  // Check `src/config/redirects.ts` to see if there is a redirect in place
  // that matches the route you are adding.
  //
  // We intentionally route most of the channel routes back to channel home
  // since we don't yet support those pages but want to gracefully redirect
  // users coming from the CDN. The real ChannelHome must come before all of the
  // placeholders to allow for proper matching behavior.
  [RouteName.ChannelHome]: '/[login]/home',
  [RouteName.ChannelCollections]: '/[login]/home',
  [RouteName.ChannelFollowers]: '/[login]/home',
  [RouteName.ChannelFollowing]: '/[login]/home',
  // Use a random string here to obfuscate access to this route
  [RouteName.ChannelStandalone]: '/[login]/kntkhbcmmyvxkmnuczri',
  [RouteName.ChannelAbout]: '/[login]/about',
  [RouteName.ChannelSchedule]: '/[login]/schedule',
  [RouteName.ChannelClips]: '/[login]/clips',
  [RouteName.ChannelVideos]: '/[login]/videos',
  [RouteName.Clip]: '/[login]/clip/[slug]',
  [RouteName.ClipLegacy]: '/clip/[slug]',
  [RouteName.External]: '/external',
  [RouteName.GameDirectory]: '/directory/game/[gameAlias]',
  [RouteName.GameDirectoryAll]: '/directory/all',
  [RouteName.GamesDirectory]: '/directory',
  [RouteName.HealthCheck]: '/_debug/running',
  [RouteName.Homepage]: '/',
  [RouteName.AppShell]: '/app-shell',
  [RouteName.NotFound]: '/not-found',
  [RouteName.ReportWizard]: '/[login]/report',
  [RouteName.Search]: '/search',
  [RouteName.Vod]: '/videos/[videoId]',
  // Channel and Upsells must be last in this list to allow for proper matching behavior
  [RouteName.Channel]: '/[login]',
  [RouteName.Upsell]: '/[login]/[upsell]',
  [RouteName.Upsell2]: '/[login]/[upsell]/[upsell2]',
};

// Converts the Next-compatible paths to standard path-to-regexp-style paths
export const RouteLinkRawPaths = Enum.entries(RouteLinkPathnames).reduce<
  Partial<RouteMap>
>((acc, [route, pathname]) => {
  acc[route] = pathname.replace(/\[(\w+)\]/g, ':$1');
  return acc;
}, {}) as RouteMap;

export type RouteLinkParams = {
  as?: string;
  href: string;
};

/**
 * A mapping of route names to a matching link builder function. The arguments
 * for these functions also serve as the definition for which "routeParams" to
 * provide to the app's "Link" component.
 *
 * DO NOT USE DIRECTLY: Use `Link` ("components/common/TachyonLink").
 */
export const RouteLinkBuilders = {
  [RouteName.Channel]: ({ login }: { login: string }): RouteLinkParams => ({
    as: `/${login}`,
    href: RouteLinkPathnames[RouteName.Channel],
  }),
  [RouteName.ChannelCollections]: ({
    login,
  }: {
    login: string;
  }): RouteLinkParams => ({
    // We intentionally route to channel home since we don't yet support this page
    as: `/${login}/home`,
    href: RouteLinkPathnames[RouteName.ChannelHome],
  }),
  [RouteName.ChannelFollowers]: ({
    login,
  }: {
    login: string;
  }): RouteLinkParams => ({
    // We intentionally route to channel home since we don't yet support this page
    as: `/${login}/home`,
    href: RouteLinkPathnames[RouteName.ChannelHome],
  }),
  [RouteName.ChannelFollowing]: ({
    login,
  }: {
    login: string;
  }): RouteLinkParams => ({
    // We intentionally route to channel home since we don't yet support this page
    as: `/${login}/home`,
    href: RouteLinkPathnames[RouteName.ChannelHome],
  }),
  [RouteName.ChannelStandalone]: ({
    login,
  }: {
    login: string;
  }): RouteLinkParams => ({
    as: `/${login}/kntkhbcmmyvxkmnuczri`,
    href: RouteLinkPathnames[RouteName.ChannelStandalone],
  }),
  [RouteName.ChannelHome]: ({ login }: { login: string }): RouteLinkParams => ({
    as: `/${login}/home`,
    href: RouteLinkPathnames[RouteName.ChannelHome],
  }),
  [RouteName.ChannelAbout]: ({
    login,
  }: {
    login: string;
  }): RouteLinkParams => ({
    as: `/${login}/about`,
    href: RouteLinkPathnames[RouteName.ChannelAbout],
  }),
  [RouteName.ChannelSchedule]: ({
    login,
  }: {
    login: string;
  }): RouteLinkParams => ({
    as: `/${login}/schedule`,
    href: RouteLinkPathnames[RouteName.ChannelSchedule],
  }),
  [RouteName.ChannelClips]: ({
    login,
  }: {
    login: string;
  }): RouteLinkParams => ({
    as: `/${login}/clips`,
    href: RouteLinkPathnames[RouteName.ChannelClips],
  }),
  [RouteName.ChannelVideos]: ({
    login,
  }: {
    login: string;
  }): RouteLinkParams => ({
    as: `/${login}/videos`,
    href: RouteLinkPathnames[RouteName.ChannelVideos],
  }),
  [RouteName.Clip]: ({
    login,
    slug,
  }: {
    login: string;
    slug: string;
  }): RouteLinkParams => ({
    as: `/${login}/clip/${slug}`,
    href: RouteLinkPathnames[RouteName.Clip],
  }),
  [RouteName.ClipLegacy]: ({ slug }: { slug: string }): RouteLinkParams => ({
    as: `/clip/${slug}`,
    href: RouteLinkPathnames[RouteName.ClipLegacy],
  }),
  // TODO: See if we can prune this out of here without upsetting our other
  // types. This will never be used in reality
  [RouteName.External]: ({ href }: { href: string }): RouteLinkParams => ({
    as: href,
    href,
  }),
  [RouteName.GameDirectory]: ({
    gameAlias,
  }: {
    gameAlias: string;
  }): RouteLinkParams => ({
    as: `/directory/game/${encodeURIComponent(gameAlias)}`,
    href: RouteLinkPathnames[RouteName.GameDirectory],
  }),
  [RouteName.GameDirectoryAll]: (): RouteLinkParams => ({
    href: RouteLinkPathnames[RouteName.GameDirectoryAll],
  }),
  [RouteName.GamesDirectory]: (): RouteLinkParams => ({
    href: RouteLinkPathnames[RouteName.GamesDirectory],
  }),
  [RouteName.HealthCheck]: (): RouteLinkParams => ({
    href: RouteLinkPathnames[RouteName.HealthCheck],
  }),
  [RouteName.Homepage]: (): RouteLinkParams => ({
    href: RouteLinkPathnames[RouteName.Homepage],
  }),
  [RouteName.AppShell]: (): RouteLinkParams => ({
    href: RouteLinkPathnames[RouteName.AppShell],
  }),
  [RouteName.NotFound]: (): RouteLinkParams => ({
    href: RouteLinkPathnames[RouteName.NotFound],
  }),
  [RouteName.ReportWizard]: ({
    login,
  }: {
    login: string;
  }): RouteLinkParams => ({
    as: `/${login}/report`,
    href: RouteLinkPathnames[RouteName.ReportWizard],
  }),
  [RouteName.Search]: (): RouteLinkParams => ({
    href: RouteLinkPathnames[RouteName.Search],
  }),
  [RouteName.Upsell]: ({
    part1,
    part2,
  }: {
    part1: string;
    part2: string;
  }): RouteLinkParams => ({
    as: `/${part1}/${part2}`,
    href: RouteLinkPathnames[RouteName.Upsell],
  }),
  [RouteName.Upsell2]: ({
    part1,
    part2,
    part3,
  }: {
    part1: string;
    part2: string;
    part3: string;
  }): RouteLinkParams => ({
    as: `/${part1}/${part2}/${part3}`,
    href: RouteLinkPathnames[RouteName.Upsell2],
  }),
  [RouteName.Vod]: ({ videoID }: { videoID: string }): RouteLinkParams => ({
    as: `/videos/${videoID}`,
    href: RouteLinkPathnames[RouteName.Vod],
  }),
};
