import { renderHook } from '@testing-library/react-hooks';
import { useRouterUtils } from 'tachyon-next-routing-utils';
import { SEARCH_TERM_QUERY_PARAM_KEY } from '../../config';
import type { AsPathForRouteNameParams } from '../linkPartsForRouteName';
import { linkPartsForRouteName } from '../linkPartsForRouteName';
import { RouteName } from '../routes';
import { mockRouter } from '../test-mocks';
import { useTachyonRouter } from '.';

jest.mock('tachyon-next-routing-utils', () => ({
  useRouterUtils: jest.fn(() => ({ lastPathname: '/' })),
}));
const mockUseRouterUtils = useRouterUtils as jest.Mock;

jest.mock('next/router', () => ({
  useRouter: () => mockRouter,
}));

describe(useTachyonRouter, () => {
  beforeEach(() => {
    mockRouter.push.mockReset();
    mockRouter.replace.mockReset();
    mockRouter.back.mockReset();
    mockRouter.reload.mockReset();
  });

  describe('push forwarding', () => {
    it('works for route without params', () => {
      const { result } = renderHook(() => useTachyonRouter());

      const route: AsPathForRouteNameParams<RouteName.HealthCheck> = {
        route: RouteName.HealthCheck,
      };

      result.current.push(route);
      const { as, href } = linkPartsForRouteName(route);
      expect(mockRouter.push).toHaveBeenCalledTimes(1);
      expect(mockRouter.push).toHaveBeenCalledWith(href, as);
    });

    it('works for route with params', () => {
      const { result } = renderHook(() => useTachyonRouter());

      const params: AsPathForRouteNameParams<RouteName.Channel> = {
        route: RouteName.Channel,
        routeParams: { login: 'lirik' },
      };

      result.current.push(params);
      const { as, href } = linkPartsForRouteName(params);
      expect(mockRouter.push).toHaveBeenCalledTimes(1);
      expect(mockRouter.push).toHaveBeenCalledWith(href, as);
    });

    it('works for route with query params', () => {
      const { result } = renderHook(() => useTachyonRouter());

      const params: AsPathForRouteNameParams<RouteName.Search> = {
        route: RouteName.Search,
      };
      const query = {
        [SEARCH_TERM_QUERY_PARAM_KEY]: 'counter',
      };

      result.current.push(params, query);
      const { as, href } = linkPartsForRouteName(params, query);
      expect(mockRouter.push).toHaveBeenCalledTimes(1);
      expect(mockRouter.push).toHaveBeenCalledWith(href, as);
    });
  });

  describe('replace forwarding', () => {
    it('works for route without params', () => {
      const { result } = renderHook(() => useTachyonRouter());

      const route: AsPathForRouteNameParams<RouteName.HealthCheck> = {
        route: RouteName.HealthCheck,
      };

      result.current.replace(route);
      const { as, href } = linkPartsForRouteName(route);
      expect(mockRouter.replace).toHaveBeenCalledTimes(1);
      expect(mockRouter.replace).toHaveBeenCalledWith(href, as);
    });

    it('works for route with params', () => {
      const { result } = renderHook(() => useTachyonRouter());

      const params: AsPathForRouteNameParams<RouteName.Channel> = {
        route: RouteName.Channel,
        routeParams: { login: 'lirik' },
      };

      result.current.replace(params);
      const { as, href } = linkPartsForRouteName(params);
      expect(mockRouter.replace).toHaveBeenCalledTimes(1);
      expect(mockRouter.replace).toHaveBeenCalledWith(href, as);
    });

    it('works for route with query params', () => {
      const { result } = renderHook(() => useTachyonRouter());

      const params: AsPathForRouteNameParams<RouteName.Search> = {
        route: RouteName.Search,
      };
      const query = {
        [SEARCH_TERM_QUERY_PARAM_KEY]: 'counter',
      };

      result.current.replace(params, query);
      const { as, href } = linkPartsForRouteName(params, query);
      expect(mockRouter.replace).toHaveBeenCalledTimes(1);
      expect(mockRouter.replace).toHaveBeenCalledWith(href, as);
    });
  });

  describe('back forwarding', () => {
    it('goes back', () => {
      const { result } = renderHook(() => useTachyonRouter());

      result.current.back();
      expect(mockRouter.back).toHaveBeenCalledTimes(1);
    });

    it('fallsback to Homepage when back would leave the app', () => {
      mockUseRouterUtils.mockImplementationOnce(() => ({
        lastPathname: undefined,
      }));
      const { result } = renderHook(() => useTachyonRouter());

      result.current.back();
      expect(mockRouter.back).toHaveBeenCalledTimes(0);
      expect(mockRouter.replace).toHaveBeenCalledWith('/');
    });
  });

  describe('reload', () => {
    it('works', () => {
      const { result } = renderHook(() => useTachyonRouter());

      result.current.reload();
      expect(mockRouter.push).toHaveBeenCalledTimes(1);
      expect(mockRouter.push).toHaveBeenCalledWith(
        mockRouter.pathname,
        mockRouter.asPath,
        { shallow: false },
      );
    });
  });
});
