import { parse } from 'url';
import accepts from 'accepts';
import type { NextFunction, RequestHandler } from 'express';
import type { ExperimentBucket } from 'tachyon-experiments';
import { selectIntlData } from 'tachyon-intl-server';
import type {
  NextRequestHandler,
  TachyonRequest,
  TachyonRequestExtension,
  TachyonResponse,
} from 'tachyon-next-types';
import { OLD_THEME_COOKIE_NAME, THEME_COOKIE_NAME } from 'tachyon-page-utils';
import {
  createDeviceIDOnServer,
  generateExperimentBucketFromDeviceID,
  getAndExtendCookieValueOnServer,
  getDeviceIDOnServer,
  getExperimentBucket,
} from 'tachyon-server-utils';
import { flattenHeaderOrParam } from 'tachyon-utils';
import type { TomorrowRequestExtensions } from '../../config';
import {
  EXPERIMENT_BUCKET_HEADER,
  LANGUAGE_QUERY_PARAM_KEY,
  THEME_QUERY_PARAM_KEY,
} from '../../config';

export const DEBUG_PATH_PREFIX = '/_debug';

export type ServerContext = Pick<
  TachyonRequestExtension<TomorrowRequestExtensions>,
  'appEnvironment' | 'dynamicSettings'
>;

export function createAppRequestHandler(
  ctx: ServerContext,
  handler: NextRequestHandler,
): RequestHandler {
  return ((
    req: TachyonRequest<TomorrowRequestExtensions>,
    res: TachyonResponse,
    _next: NextFunction,
  ): void => {
    if (process.env.NODE_ENV !== 'test') {
      console.log(`# Handling request for ${req.path}`);
    }

    const headerLanguages = accepts(req).languages();
    const requestLanguageQueryParam = flattenHeaderOrParam(
      req.query[LANGUAGE_QUERY_PARAM_KEY],
    );

    const userPreferredLanguageTags = requestLanguageQueryParam
      ? [requestLanguageQueryParam, ...headerLanguages]
      : headerLanguages;

    const intlData = selectIntlData(userPreferredLanguageTags);

    // theme query param overrides theme cookie
    const initialTheme =
      (req.query[THEME_QUERY_PARAM_KEY] as string | undefined) ??
      getAndExtendCookieValueOnServer({
        migrationNames: [OLD_THEME_COOKIE_NAME],
        name: THEME_COOKIE_NAME,
        req,
        res,
      });

    let deviceId: string | undefined;
    let experimentBucket: ExperimentBucket | undefined;
    // disregard debug paths since they aren't CDN-fronted and thus won't have user info
    if (!req.path.startsWith(DEBUG_PATH_PREFIX)) {
      // Ideally both deviceID and experimentBucket always defined in production
      // since they should be managed by Fastly, and thus optimistically this
      // fallback logic only exists for local dev UX.
      deviceId = getDeviceIDOnServer(req, res) ?? createDeviceIDOnServer(res);
      experimentBucket =
        getExperimentBucket(req, {
          header: EXPERIMENT_BUCKET_HEADER,
          package: 'tomorrow',
        }) ?? generateExperimentBucketFromDeviceID(deviceId);
    }

    req.tachyon = {
      ...ctx,
      deviceId,
      experimentBucket,
      initialTheme,
      intlData,
    };

    handler(req, res, parse(req.url, true));
    // cast to narrow back down to express types
  }) as any as RequestHandler;
}
