import { parse } from 'url';
import { datatype, lorem, random } from 'faker';
import type { IntlData } from 'tachyon-intl';
import { selectIntlData } from 'tachyon-intl-server';
import type {
  TachyonRequest,
  TachyonRequestExtension,
  TachyonResponse,
} from 'tachyon-next-types';
import { THEME_COOKIE_NAME } from 'tachyon-page-utils';
import { createDeviceIDOnServer } from 'tachyon-server-utils';
import { GLOBAL_COOKIE_NAMES, TACHYON_DEVICE_ID_PREFIX } from 'tachyon-utils';
import type { TomorrowRequestExtensions } from '../../config';
import {
  EXPERIMENT_BUCKET_HEADER,
  EXPERIMENT_OVERRIDES_QUERY_PARAM_KEY,
  LANGUAGE_QUERY_PARAM_KEY,
  THEME_QUERY_PARAM_KEY,
} from '../../config';
import { DEBUG_PATH_PREFIX, createAppRequestHandler } from '.';

const mockLocale = random.locale();
jest.mock('accepts', () => () => ({
  languages: () => [mockLocale],
}));

const mockIntlData: IntlData = {
  locales: [],
  preferredLanguageTags: [mockLocale],
};
jest.mock('tachyon-intl-server', () => ({
  selectIntlData: jest.fn(() => mockIntlData),
}));
const mockSelectIntlData = selectIntlData as jest.Mock;

jest.mock('tachyon-server-utils', () => ({
  ...jest.requireActual('tachyon-server-utils'),
  createDeviceIDOnServer: jest.fn(() => 'tach-fake-device-id'),
}));
const mockCreateDeviceIDOnServer = createDeviceIDOnServer as jest.Mock;

describe(createAppRequestHandler, () => {
  const mockContext = {
    appEnvironment: 'development',
    dynamicSettings: {},
  } as TachyonRequestExtension;
  const deviceId = datatype.uuid();
  const path = `/${lorem
    .words(datatype.number({ max: 4 }))
    .replace(/ /g, '/')}`;
  const experimentBucket = 'f';
  const experimentOverrides = datatype.uuid();

  let req: TachyonRequest<TomorrowRequestExtensions>;
  let res: TachyonResponse;

  const next = () => undefined;
  const mockNextRequestHandler = jest.fn();
  const handler = createAppRequestHandler(mockContext, mockNextRequestHandler);

  beforeEach(() => {
    mockNextRequestHandler.mockReset();

    req = {
      cookies: {
        [GLOBAL_COOKIE_NAMES.DEVICE_ID_COOKIE_NAME]: deviceId,
      },
      headers: {
        [EXPERIMENT_BUCKET_HEADER]: experimentBucket,
      },
      path,
      query: {
        [EXPERIMENT_OVERRIDES_QUERY_PARAM_KEY]: experimentOverrides,
      },
      url: path,
    } as any;

    res = { cookie: jest.fn() } as any;
  });

  describe('when the lang query param is present', () => {
    it('select the locale bundle using the lang query param value', () => {
      const languageQueryParamValue = 'fr';
      req.query[LANGUAGE_QUERY_PARAM_KEY] = languageQueryParamValue;

      handler(req, res, next);
      expect(mockSelectIntlData).toHaveBeenLastCalledWith([
        languageQueryParamValue,
        mockLocale,
      ]);
    });
  });

  it('creates a request handler function that adds static properties to req and calls next request handler', () => {
    expect(handler).toBeInstanceOf(Function);

    handler(req, res, next);
    expect(req.tachyon).toEqual(
      expect.objectContaining({
        appEnvironment: 'development',
        dynamicSettings: expect.objectContaining({}),
        intlData: mockIntlData,
      }),
    );

    expect(mockSelectIntlData).toHaveBeenLastCalledWith([mockLocale]);
    expect(mockNextRequestHandler).toHaveBeenCalledWith(
      req,
      res,
      parse(req.url, true),
    );
  });

  describe('initial theme', () => {
    it('passes through initial theme from query param', () => {
      req.query[THEME_QUERY_PARAM_KEY] = 'light';
      handler(req, res, next);
      expect(req.tachyon.initialTheme).toEqual('light');
    });

    it('passes through initial theme from cookie', () => {
      req.cookies[THEME_COOKIE_NAME] = 'light';
      handler(req, res, next);
      expect(req.tachyon.initialTheme).toEqual('light');
    });

    it('passes through initial theme from query param overriding cookie', () => {
      req.query[THEME_QUERY_PARAM_KEY] = 'light';
      req.cookies[THEME_COOKIE_NAME] = 'dark';
      handler(req, res, next);
      expect(req.tachyon.initialTheme).toEqual('light');
    });

    it('is undefined without query param or cookie', () => {
      handler(req, res, next);
      expect(req.tachyon.initialTheme).toBeUndefined();
    });
  });

  describe('device id and experiment bucket', () => {
    it('finds an existing device id and experiment bucket and does not warn', () => {
      handler(req, res, next);
      expect(req.tachyon.deviceId).toEqual(deviceId);

      expect(req.tachyon).toEqual(
        expect.objectContaining({
          deviceId,
          experimentBucket,
        }),
      );

      expect(mockCreateDeviceIDOnServer).not.toHaveBeenCalled();
    });

    it('creates a missing device id when only experiment bucket is found and warns', () => {
      delete req.cookies[GLOBAL_COOKIE_NAMES.DEVICE_ID_COOKIE_NAME];
      handler(req, res, next);

      expect(req.tachyon).toEqual(
        expect.objectContaining({
          deviceId: expect.stringContaining(TACHYON_DEVICE_ID_PREFIX),
          experimentBucket,
        }),
      );
      expect(mockCreateDeviceIDOnServer).toHaveBeenCalled();
    });

    it('creates a missing experiment bucket when only device ID is found and warns', () => {
      delete req.headers[EXPERIMENT_BUCKET_HEADER];
      handler(req, res, next);

      expect(req.tachyon).toEqual(
        expect.objectContaining({
          deviceId,
          experimentBucket: expect.any(String),
        }),
      );
      expect(mockCreateDeviceIDOnServer).not.toHaveBeenCalled();
    });

    it('creates a missing experiment bucket and device ID and warns', () => {
      delete req.cookies[GLOBAL_COOKIE_NAMES.DEVICE_ID_COOKIE_NAME];
      delete req.headers[EXPERIMENT_BUCKET_HEADER];
      handler(req, res, next);

      expect(req.tachyon).toEqual(
        expect.objectContaining({
          deviceId: expect.any(String),
          experimentBucket: expect.any(String),
        }),
      );
      expect(mockCreateDeviceIDOnServer).toHaveBeenCalled();
    });

    it('does nothing for debug paths', () => {
      delete req.cookies[GLOBAL_COOKIE_NAMES.DEVICE_ID_COOKIE_NAME];
      delete req.headers[EXPERIMENT_BUCKET_HEADER];
      req.path = DEBUG_PATH_PREFIX;

      handler(req, res, next);

      expect(req.tachyon).toEqual(
        expect.not.objectContaining({
          deviceId: expect.anything(),
          experimentBucket: expect.anything(),
        }),
      );
      expect(mockCreateDeviceIDOnServer).not.toHaveBeenCalled();
    });
  });
});
