import { resolve } from 'path';
import { URL } from 'url';
import cookieParser from 'cookie-parser';
import express from 'express';
import next from 'next';
import open from 'open';
import { fetchDynamicSettings } from 'tachyon-dynamic-settings';
import { prepareIntlDataCache } from 'tachyon-intl-server';
import {
  getTachyonEnvVar,
  setCookieDomainOnServer,
  tachyonServerMiddleware,
} from 'tachyon-server-utils';
import {
  ensureString,
  ensureStringArray,
  getCurrentTwitchDomain,
} from 'tachyon-utils';
import type { TomorrowDynamicSettings } from '../config';
import {
  BGSYNC_QUERY_PARAM_KEY,
  BGSYNC_QUERY_PARAM_VALUE,
  CHANNEL_ALIAS_MIDDLEWARE_OPTS,
  FLUME_ALLOWED_EVENTS_FALLBACK,
  FLUME_ALLOWED_PROPERTIES_FALLBACK,
  PAGE_ALIAS_MIDDLEWARE_OPTS,
  REDIRECT_TO_DESKTOP_WEB_MIDDLEWARE_OPTS,
} from '../config';
import { RouteName, dynamicPathnames, pathnameFromRouteName } from '../routing';
import { createAppRequestHandler } from './appHandler';
import { isDevelopment } from './utils';

// polyfill WHATWG URL
// eslint-disable-next-line @typescript-eslint/no-unsafe-member-access
(global as any).URL = URL;

const PORT = process.env.PORT ? parseInt(process.env.PORT, 10) : 3200;
const dev = isDevelopment();
// from `dist/server/index.js` to dir containing `.next/` & `messages/`
const appDir = dev ? undefined : resolve(__dirname, '../..');

// 15 minutes
const DYNAMIC_SETTINGS_TTL = 15 * 60 * 1000;

const appEnvironment = getTachyonEnvVar();

// istanbul ignore next: trivial + type-based enforcement
const fetchTomorrowDynamicSettings =
  fetchDynamicSettings<TomorrowDynamicSettings>({
    app: 'tomorrow',
    appEnvironment,
    appGroup: 'tachyon',
    processor: ({
      cookie_notice_country_codes,
      cookie_policy_url,
      flume_allowed_events,
      flume_allowed_properties,
      privacy_policy_url,
      spadeUrl,
      terms_of_service_url,
      ...settings
    }) => {
      // the base dynamic settings processor guarantees returning a valid URI,
      // so we don't need to try-catch
      const customSpadeUrl = new URL(spadeUrl);
      customSpadeUrl.searchParams.append(
        BGSYNC_QUERY_PARAM_KEY,
        BGSYNC_QUERY_PARAM_VALUE,
      );

      const cookieNoticeCountryCodes = ensureStringArray(
        cookie_notice_country_codes,
        [],
      );
      const cookiePolicyUrl = ensureString(cookie_policy_url, '');
      const flumeAllowedEvents = ensureStringArray(
        flume_allowed_events,
        FLUME_ALLOWED_EVENTS_FALLBACK,
      );
      const flumeAllowedProperties = ensureStringArray(
        flume_allowed_properties,
        FLUME_ALLOWED_PROPERTIES_FALLBACK,
      );
      const privacyPolicyUrl = ensureString(privacy_policy_url, '');
      const termsOfServiceUrl = ensureString(terms_of_service_url, '');

      return {
        ...settings,
        cookieNoticeCountryCodes,
        cookiePolicyUrl,
        flumeAllowedEvents,
        flumeAllowedProperties,
        privacyPolicyUrl,
        spadeUrl: customSpadeUrl.toString(),
        termsOfServiceUrl,
      };
    },
  });

// top-level await unavailable until Node 14.8
// istanbul ignore next: trivial
prepareIntlDataCache({
  appDir,
  includePackages: ['tachyon-chat-ui', 'tachyon-more-ui'],
})
  .then(async () => {
    const dynamicSettings = await fetchTomorrowDynamicSettings();
    const ctx = {
      appEnvironment,
      dynamicSettings,
    };

    setInterval(() => {
      fetchTomorrowDynamicSettings().then((settings) => {
        ctx.dynamicSettings = settings;
      });
    }, DYNAMIC_SETTINGS_TTL).unref();

    return ctx;
  })
  .then((ctx) => {
    const app = next({ dev, dir: appDir });
    const nextRequestHandler = app.getRequestHandler();
    const appRequestHandler = createAppRequestHandler(ctx, nextRequestHandler);

    return app.prepare().then(() => {
      const server = express();
      server.use(
        tachyonServerMiddleware({
          badRequestMiddleware: {
            badRequestRedirectPath: pathnameFromRouteName(RouteName.NotFound),
            dynamicPathnames: dynamicPathnames(),
          },
          channelAliasMiddleware: CHANNEL_ALIAS_MIDDLEWARE_OPTS,
          pageAliasMiddleware: PAGE_ALIAS_MIDDLEWARE_OPTS,
          redirectToDesktopWebMiddleware:
            REDIRECT_TO_DESKTOP_WEB_MIDDLEWARE_OPTS,
        }),
      );
      server.use(cookieParser());
      // tell express to populate `req` convenience attributes from X-Forwarded-* headers
      // http://expressjs.com/en/4x/api.html#req.hostname
      server.enable('trust proxy');

      // api routes are the only routes that need to handle POST requests, so we
      // make this explicit to minimize our exposed surface area
      server.post('/api/*', appRequestHandler);
      server.get('*', appRequestHandler);

      // allow setting alternate cookie domains (for staging environments only)
      const domain = getCurrentTwitchDomain(process.env.TACHYON_COOKIE_DOMAIN);
      if (domain && getTachyonEnvVar() === 'staging') {
        setCookieDomainOnServer(domain);
      }

      server.listen(PORT, () => {
        if (dev) {
          const devUrl = 'https://localhost.m.twitch.tv';
          console.log(`\nListening at ${devUrl}\n`);
          open(devUrl);
        }
      });
    });
  })
  .catch((error: Error) => {
    console.error(error.stack);
    process.exit(1);
  });
