import Head from 'next/head';
import type { FC } from 'react';
import { useCallback, useMemo, useState } from 'react';
import { createGlobalStyle } from 'styled-components';
import { CoreUiSsrRoot, tachyonStyleOverrides } from 'tachyon-more-ui';
import { setThemeCookie } from 'tachyon-page-utils';
import { DynamicThemeMap } from 'twitch-core-ui-tokens';
import type { Theme } from '../Theme';
import { convertPreferredThemeToBaseTheme } from '../Theme';
import { themeContext } from '../ThemeContext';
import { usePreferredTheme } from '../usePreferredTheme';
import { validateTheme } from '../validateTheme';

// istanbul ignore next: trivial
export const GlobalStyle = createGlobalStyle<{ background: string }>`
  ${tachyonStyleOverrides}
  html, body {
    background: ${({ background }) => background};
  }
`;

export interface ThemeRootProps {
  initialTheme: string | undefined;
}

/**
 * Adds theme control as a stopgap until TMW gets more official support from a
 * newer version of Core UI. This takes an initial theme (derived from a
 * cookie), but controls itself after that. The initial theme value can signify
 * three states: explicit user selection, device preference indication, or no
 * preference of any type recorded.
 *
 * If the initial theme indicates an explicit user selection, that choice will
 * persist despite device preference (which will no longer be queried).
 * Otherwise, this component will query the CSSOM for prefers-color-scheme
 * support and preference, and if it detects a supported theme preference it
 * will set a theme cookie with a preferred value to limit flashing on any
 * subsequent loads. Even with a initial theme representing a device preference,
 * it will requery again on next boot in case that setting has changed. As a
 * fallback, no initial theme (no cookie) and no device preference indication
 * results in using the default theme. If we're relying on device preference, we
 * watch for changes and update accordingly.
 *
 * Toggling the theme via the UI sets/updates the theme cookie with the explicit
 * user choice.
 */
export const ThemeRoot: FC<ThemeRootProps> = ({ children, initialTheme }) => {
  const [theme, setThemeState] = useState(() => validateTheme(initialTheme));
  const setTheme = useCallback((nextTheme: Theme) => {
    setThemeCookie(nextTheme);
    setThemeState(nextTheme);
  }, []);

  usePreferredTheme(theme, setTheme);

  const currentTheme = convertPreferredThemeToBaseTheme(theme);

  const toggleTheme = useCallback(() => {
    const nextTheme = currentTheme === 'dark' ? 'light' : 'dark';
    setTheme(nextTheme);
  }, [currentTheme, setTheme]);

  const ctx = useMemo(
    () => ({ currentTheme, toggleTheme }),
    [currentTheme, toggleTheme],
  );

  return (
    <>
      <Head>
        <meta
          content={DynamicThemeMap[currentTheme]['color-background-base']}
          name="theme-color"
        />
        {/* This value is only checked once, on A2HS installation. Toggling this value in app does not update the status bar. Because of that, we set to 'default' which will set the status bar light or dark based on the iOS theme. Enabling Dark Mode on iOS will make the status bar dark. NOTE: The app needs to be closed and relaunched to pick up OS theme changes. */}
        <meta content="default" name="apple-mobile-web-app-status-bar-style" />
      </Head>
      <GlobalStyle
        background={DynamicThemeMap[currentTheme]['color-background-body']}
      />
      <themeContext.Provider value={ctx}>
        {/* eslint-disable-next-line react/forbid-dom-props */}
        <div className={`tw-root--theme-${currentTheme}`}>
          <CoreUiSsrRoot
            appRootElementId="__next"
            disableHoverCSS
            theme={currentTheme}
          >
            {children}
          </CoreUiSsrRoot>
        </div>
      </themeContext.Provider>
    </>
  );
};

ThemeRoot.displayName = 'ThemeRoot';
