import { useEffect } from 'react';
import { useEffectOnce } from 'tachyon-utils';
import type { SetTheme, Theme } from '../Theme';

const lightMediaQuery = '(prefers-color-scheme: light)';
const darkMediaQuery = '(prefers-color-scheme: dark)';

// istanbul ignore next: trivial
export function useInitialPreferredTheme(
  currentTheme: Theme,
  setTheme: SetTheme,
): void {
  useEffectOnce(() => {
    if (currentTheme.endsWith('preferred')) {
      if (window.matchMedia(lightMediaQuery).matches) {
        setTheme('light_preferred');
      }

      if (window.matchMedia(darkMediaQuery).matches) {
        setTheme('dark_preferred');
      }
    }
  });
}

// istanbul ignore next: trivial to test in browser, brittle to mock
export function useWatchPreferredTheme(
  currentTheme: Theme,
  setTheme: SetTheme,
): void {
  useEffect(() => {
    const lightMatcher = window.matchMedia(lightMediaQuery);
    const lightSwitcher = (e: MediaQueryListEvent) => {
      if (e.matches && currentTheme.endsWith('preferred')) {
        setTheme('light_preferred');
      }
    };
    // matchesMedia originally had a custom `addListener` version that is now
    // deprecated in iOS 14; we don't support the old API but do protect against
    // error
    // https://bugs.webkit.org/show_bug.cgi?id=203288 for instance
    // eslint-disable-next-line @typescript-eslint/no-unnecessary-condition, @typescript-eslint/no-confusing-void-expression
    lightMatcher.addEventListener?.('change', lightSwitcher);

    const darkMatcher = window.matchMedia(darkMediaQuery);
    const darkSwitcher = (e: MediaQueryListEvent) => {
      if (e.matches && currentTheme.endsWith('preferred')) {
        setTheme('dark_preferred');
      }
    };
    // eslint-disable-next-line @typescript-eslint/no-unnecessary-condition, @typescript-eslint/no-confusing-void-expression
    darkMatcher.addEventListener?.('change', darkSwitcher);

    return () => {
      // eslint-disable-next-line @typescript-eslint/no-unnecessary-condition, @typescript-eslint/no-confusing-void-expression
      lightMatcher.removeEventListener?.('change', lightSwitcher);
      // eslint-disable-next-line @typescript-eslint/no-unnecessary-condition, @typescript-eslint/no-confusing-void-expression
      darkMatcher.removeEventListener?.('change', darkSwitcher);
    };
  }, [currentTheme, setTheme]);
}

// istanbul ignore next: trivial
export function usePreferredTheme(
  currentTheme: Theme,
  setTheme: SetTheme,
): void {
  useInitialPreferredTheme(currentTheme, setTheme);
  useWatchPreferredTheme(currentTheme, setTheme);
}
