import type { AgentInfo } from 'tachyon-environment';
import { isAndroidOS, isFireOS, isIOS } from 'tachyon-environment';

const IOS_STORE_APP_ID = 'id460177396';
const PLAY_STORE_APP_ID = 'tv.twitch.android.app';
const AMAZON_APP_STORE_APP_ID = 'tv.twitch.android.viewer';

export enum DeeplinkableOS {
  Android = 'Android',
  Fire = 'Fire',
  iOS = 'iOS',
}

interface AppStoreOpts {
  callToActionIdentifier: string;
  campaign: string;
  os: DeeplinkableOS;
}

interface DeepLinkOpts {
  appStoreOpts?: AppStoreOpts;
  path: string;
  referrer: string;
}

export function getDeeplinkableOS(agent: AgentInfo): DeeplinkableOS | null {
  if (isIOS(agent)) {
    return DeeplinkableOS.iOS;
  }

  if (isFireOS(agent)) {
    return DeeplinkableOS.Fire;
  }

  if (isAndroidOS(agent)) {
    return DeeplinkableOS.Android;
  }

  return null;
}

export function buildDeepLinkURL(opts: DeepLinkOpts): string {
  if (!opts.appStoreOpts) {
    return `twitch://m.twitch.tv${opts.path}?referrer=${encodeURIComponent(
      opts.referrer,
    )}`;
  }
  switch (opts.appStoreOpts.os) {
    case DeeplinkableOS.iOS:
      return buildiOSDeepLinkURL(
        opts.appStoreOpts.campaign,
        opts.appStoreOpts.callToActionIdentifier,
      );
    case DeeplinkableOS.Android:
      return buildAndroidDeepLinkURL(
        opts.referrer,
        opts.path,
        opts.appStoreOpts.campaign,
        PLAY_STORE_APP_ID,
      );
    case DeeplinkableOS.Fire:
      return buildAndroidDeepLinkURL(
        opts.referrer,
        opts.path,
        opts.appStoreOpts.campaign,
        AMAZON_APP_STORE_APP_ID,
        false,
      );
    // This is logically impossible due to our type system
    default:
      return undefined as never;
  }
}

// not technically a deeplink, but it's the best we can do
function buildiOSDeepLinkURL(
  campaign: string,
  callToActionIdentifier: string,
): string {
  return `https://apps.apple.com/app/twitch-live-game-streaming/${IOS_STORE_APP_ID}?pt=325952&mt=8&ct=${campaign}-${callToActionIdentifier}`;
}

function buildAndroidDeepLinkURL(
  referrer: string,
  path: string,
  campaign: string,
  appID: string,
  haveFallback = true,
): string {
  // doubly encode the referrer to avoid break on & in urls params, etc
  const encodedMarketURL = encodeURIComponent(`market://details?id=${appID}`);
  // There seems to be a weird Chrome bug where if it goes source then medium then the params
  // passed to the play store cause it to not load the Twitch app up in the store if the app
  // was previously running. However, putting them in this order makes it work fine.
  const encodedMarketReferrer = encodeURIComponent(
    `utm_campaign=${encodeURIComponent(campaign)}&utm_source=mwc`,
  );

  return (
    `intent://m.twitch.tv${path}?referrer=${encodeURIComponent(referrer)}` + // eslint-disable-line prefer-template
    `#Intent;package=${appID};scheme=twitch;` +
    `S.market_referrer=${encodedMarketReferrer};` +
    (haveFallback ? `S.browser_fallback_url=${encodedMarketURL};` : '') +
    'end;'
  );
}
