import { MockUserAgent, getAgentInfo } from 'tachyon-environment';
import { DeeplinkableOS, buildDeepLinkURL, getDeeplinkableOS } from '.';

interface DeepLinkableOsTestCase {
  agent: string;
  expected: DeeplinkableOS | null;
}

describe(buildDeepLinkURL, () => {
  it('builds with a custom intent', () => {
    expect(
      buildDeepLinkURL({ path: '/directory/bigandy', referrer: 'hi' }),
    ).toEqual('twitch://m.twitch.tv/directory/bigandy?referrer=hi');
  });

  it('encodes the referrer', () => {
    expect(
      buildDeepLinkURL({ path: '/a', referrer: 'https://m.twitch.tv?h=i' }),
    ).toEqual(
      'twitch://m.twitch.tv/a?referrer=https%3A%2F%2Fm.twitch.tv%3Fh%3Di',
    );
  });

  it('builds an android deep link', () => {
    expect(
      buildDeepLinkURL({
        appStoreOpts: {
          callToActionIdentifier: 'a_button',
          campaign: 'some_campaign',
          os: DeeplinkableOS.Android,
        },
        path: '/monstercat',
        referrer: 'ref',
      }),
    ).toEqual(
      'intent://m.twitch.tv/monstercat?referrer=ref#Intent;package=tv.twitch.android.app;scheme=twitch;S.market_referrer=utm_campaign%3Dsome_campaign%26utm_source%3Dmwc;S.browser_fallback_url=market%3A%2F%2Fdetails%3Fid%3Dtv.twitch.android.app;end;',
    );
  });

  it('builds a Fire deep link', () => {
    expect(
      buildDeepLinkURL({
        appStoreOpts: {
          callToActionIdentifier: 'a_button',
          campaign: 'some_campaign',
          os: DeeplinkableOS.Fire,
        },
        path: '/monstercat',
        referrer: 'ref',
      }),
    ).toEqual(
      'intent://m.twitch.tv/monstercat?referrer=ref#Intent;package=tv.twitch.android.viewer;scheme=twitch;S.market_referrer=utm_campaign%3Dsome_campaign%26utm_source%3Dmwc;end;',
    );
  });

  it('builds an iOS deep link', () => {
    expect(
      buildDeepLinkURL({
        appStoreOpts: {
          callToActionIdentifier: 'b_button',
          campaign: 'some_campaign',
          os: DeeplinkableOS.iOS,
        },
        path: '/monsterdog',
        referrer: 'ref',
      }),
    ).toEqual(
      'https://apps.apple.com/app/twitch-live-game-streaming/id460177396?pt=325952&mt=8&ct=some_campaign-b_button',
    );
  });
});

describe(getDeeplinkableOS, () => {
  it.each`
    agent                                     | expected
    ${MockUserAgent.iOS[9].iPhone}            | ${DeeplinkableOS.iOS}
    ${MockUserAgent.Android['6'].NexusTablet} | ${DeeplinkableOS.Android}
    ${MockUserAgent.Silk}                     | ${DeeplinkableOS.Fire}
    ${MockUserAgent.macOS.Safari}             | ${null}
    ${MockUserAgent.WindowsPhone}             | ${null}
  `(
    'returns $expected for $agent',
    ({ agent, expected }: DeepLinkableOsTestCase) => {
      expect(getDeeplinkableOS(getAgentInfo(agent))).toEqual(expected);
    },
  );
});
