/* eslint-disable import/order*/
const { ProvidePlugin } = require('webpack');
const SentryWebpackPlugin = require('@sentry/webpack-plugin');
const withFonts = require('next-fonts');
const withBundleAnalyzer = require('@next/bundle-analyzer')({
  enabled: !!process.env.ANALYZE,
});
const withSourceMaps = require('@zeit/next-source-maps')({
  devtool: process.env.DOCKER_PRODUCTION ? 'hidden-source-map' : 'source-map',
});
const { buildId } = require('./env.config');
const { ALLOWED_FRAME_ANCESTORS, REDIRECT_TO_DESKTOP_WEB } =
  process.env.NODE_ENV === 'production'
    ? require('./dist/config')
    : require('./src/config');

// Valence next config
const NEXT_CONFIG = {
  webpack5: false,
  // ignore typescript errors in local dev
  typescript: {
    ignoreDevErrors: true,
  },

  // set the build ID to the git hash for determinism
  generateBuildId: () => buildId,

  async headers() {
    return [
      {
        source: '/(.*)',
        headers: [
          {
            key: 'Content-Security-Policy',
            value: `frame-ancestors ${ALLOWED_FRAME_ANCESTORS}`,
          },
        ],
      },
    ];
  },

  async redirects() {
    return [
      // https://github.com/vercel/next.js/issues/17189
      {
        source: '/_error',
        destination: '/not-found',
        permanent: false,
      },
      ...REDIRECT_TO_DESKTOP_WEB.map((source) => ({
        source,
        destination: `https://www.twitch.tv${source}`,
        permanent: false,
      })),
    ];
  },

  // custom webpack config
  webpack(config, { defaultLoaders, dev, isServer }) {
    // ensure that bundle order doesn't result in a regeneratorRuntime error
    config.plugins.push(
      new ProvidePlugin({
        regeneratorRuntime: 'regenerator-runtime/runtime',
      }),
    );

    if (!isServer) {
      config.plugins.push(
        new SentryWebpackPlugin({
          dryRun: !process.env.SENTRY_AUTH_TOKEN,
          include: '.next',
          release: buildId,
          urlPrefix: '~/_next',
        }),
      );
    }

    // set up aliases
    const alias = config.resolve.alias || {};

    // react-router isn't used by Tachyon apps but is required by core-ui;
    // only necessary because of unused transient dependencies related to
    // latency components that aren't tree-shaken on the server
    config.resolve.alias = {
      ...alias,
      'react-router': 'tachyon-noop',
      'react-router-dom': 'tachyon-noop',
    };

    return config;
  },
};

module.exports = (phase, { defaultConfig }) => {
  const baseConfig = withSourceMaps(withFonts(NEXT_CONFIG));

  if (process.env.DOCKER_PRODUCTION) {
    return baseConfig;
  }

  return withBundleAnalyzer(baseConfig);
};
