import { usePlayerController } from 'pulsar';
import type { FC } from 'react';
import { useEffect, useRef } from 'react';
import { useEffectOnce } from 'tachyon-utils';
import { RouteName, useTachyonRouter } from '../../../routing';
import type { ControllerContext } from '../ControllerContext';
import { controllerContext } from '../ControllerContext';
import type { LoadedEventData } from '../types';
import { handleCommand, sendEvent } from '../utils';

function onLoad(metadata: LoadedEventData): void {
  sendEvent('loaded', metadata);
}

export type ControllerRootProps = {
  isUnauthorizedAccess: boolean;
};

/**
 * The Controller functionality allows the iframed app to be controlled by a
 * parent via the PostMessage API.
 */
export const ControllerRoot: FC<ControllerRootProps> = ({
  children,
  isUnauthorizedAccess,
}) => {
  const router = useTachyonRouter();
  const playerController = usePlayerController();

  const { current: context } = useRef<ControllerContext>({
    onLoad,
    onPlayerEvent: sendEvent,
  });

  useEffectOnce(() => {
    // if unauthorized, redirect to blank page
    if (isUnauthorizedAccess) {
      router.push({ route: RouteName.NotFound });
      return;
    }

    sendEvent('init');
  });

  useEffect(() => {
    const onCommand = handleCommand(playerController, router);
    window.addEventListener('message', onCommand);

    return () => {
      window.removeEventListener('message', onCommand);
    };
  }, [playerController, router]);

  return <controllerContext.Provider children={children} value={context} />;
};

ControllerRoot.displayName = 'ControllerRoot';
