type SimpleCommand = {
  command: 'getMuted' | 'getPosition' | 'getVolume' | 'pause' | 'play' | 'stop';
};

type LoadCommandDataOptions = {
  /**
   * Playback defaults to autoPlay.
   *
   * When true, content will automatically begin playing when loaded. When false, content will remain in the paused state until started.
   */
  autoPlay?: boolean;
  /**
   * Float: [0.0, 1.0].
   */
  initialVolume?: boolean;
  /**
   * Playback defaults to muted.
   */
  startMuted?: boolean;
};

type LoadCommandDataBase = {
  id: string;
  options?: LoadCommandDataOptions;
};

type LoadCommandDataStream = LoadCommandDataBase & {
  type: 'stream';
};

type LoadCommandDataVod = LoadCommandDataBase & {
  options?: LoadCommandDataOptions & {
    /**
     * Float: [0.0, VideoDuration] (seconds)
     */
    startFromTime?: number;
  };
  type: 'vod';
};

type LoadCommandDataClip = LoadCommandDataBase & {
  options?: LoadCommandDataOptions & {
    /**
     * Playback will loop by default.
     */
    loop?: boolean;
  };
  type: 'clip';
};

type LoadCommandData =
  | LoadCommandDataClip
  | LoadCommandDataStream
  | LoadCommandDataVod;

type LoadCommand = {
  command: 'load';
  data: LoadCommandData;
};

type SetMutedCommand = {
  command: 'setMuted';
  data: boolean;
};

type SetVolumeCommand = {
  command: 'setVolume';
  /**
   * Float: [0.0, 1.0]
   */
  data: number;
};

type SetPositionCommand = {
  command: 'setPosition';
  /**
   * Float: [0.0, VideoDuration] (seconds)
   */
  data: number;
};

export type InboundCommand =
  | LoadCommand
  | SetMutedCommand
  | SetPositionCommand
  | SetVolumeCommand
  | SimpleCommand;

export type LoadedEventData =
  | {
      /**
       * Start time in simplified extended ISO format (ISO 8601)
       */
      broadcastTimestamp: string | null;
      duration: number | null;
      type: 'clip' | 'vod';
    }
  | {
      broadcastTimestamp: string | null;
      type: 'stream';
    };

type LoadedEvent = {
  data: LoadedEventData;
  event: 'loaded';
};

type SimpleEvent = {
  event:
    | 'autoplayFailed'
    | 'bufferEmptied'
    | 'bufferRefilled'
    | 'contentEnded'
    | 'init'
    | 'paused'
    | 'played'
    | 'stopped';
};

type MutedStatusEvent = {
  data: boolean;
  event: 'mutedStatus';
};

type MutedSetEvent = {
  data: boolean;
  event: 'mutedSet';
};

type PositionStatusEvent = {
  data: number;
  event: 'positionStatus';
};

type PositionSetEvent = {
  data: number;
  event: 'positionSet';
};

type VolumeStatusEvent = {
  data: number;
  event: 'volumeStatus';
};

type VolumeSetEvent = {
  data: number;
  event: 'volumeSet';
};

type SeekCompletedEvent = {
  /**
   * The position where the seek completed, in seconds.
   */
  data: number;
  event: 'seekCompleted';
};

/**
 * Indicates a fatal error occurred. The iframe will not respond to subsequent commands and must be
 * reloaded.
 */
type AppErroredEvent = {
  event: 'appErrored';
};

/**
 * A video playback error occurred. A new play session may be started by sending a "load" command.
 */
type VideoErroredEvent = {
  event: 'videoErrored';
};

export type OutboundEvent =
  | AppErroredEvent
  | LoadedEvent
  | MutedSetEvent
  | MutedStatusEvent
  | PositionSetEvent
  | PositionStatusEvent
  | SeekCompletedEvent
  | SimpleEvent
  | VideoErroredEvent
  | VolumeSetEvent
  | VolumeStatusEvent;
