import type { PlayerController } from 'pulsar';
import {
  AUTOPLAY_ON,
  AUTOPLAY_PARAM,
  DEFAULT_INITIAL_VOLUME,
  INITIAL_MUTED_STATE_PARAM,
  INITIAL_VOLUME_PARAM,
  LOOP_ON,
  LOOP_PARAM,
  START_FROM_TIME_PARAM,
  START_MUTED,
} from '../../../config';
import type { TachyonRouter } from '../../../routing';
import { RouteName } from '../../../routing';
import { isAllowedOrigin } from './isAllowedOrigin';
import { parseControllerCommand } from './parseControllerCommand';
import { sendEvent } from './sendEvent';

export const handleCommand =
  (playerController: PlayerController | null, router: TachyonRouter) =>
  ({ data: dataString, origin }: MessageEvent): void => {
    if (!isAllowedOrigin(origin)) {
      return;
    }

    const payload = parseControllerCommand(dataString);
    if (!payload) {
      return;
    }

    // navigation-based commands
    if (payload.command === 'load') {
      const baseParams = {
        [AUTOPLAY_PARAM]:
          payload.data.options?.autoPlay?.toString() ?? AUTOPLAY_ON,
        [INITIAL_MUTED_STATE_PARAM]:
          payload.data.options?.startMuted?.toString() ?? START_MUTED,
        [INITIAL_VOLUME_PARAM]:
          payload.data.options?.initialVolume?.toString() ??
          DEFAULT_INITIAL_VOLUME,
      };

      switch (payload.data.type) {
        case 'stream':
          router.push(
            {
              route: RouteName.Channel,
              routeParams: { login: payload.data.id },
            },
            baseParams,
          );
          break;
        case 'vod':
          router.push(
            {
              route: RouteName.Vod,
              routeParams: { videoId: payload.data.id },
            },
            {
              ...baseParams,
              [START_FROM_TIME_PARAM]:
                payload.data.options?.startFromTime?.toString() ?? '',
            },
          );
          break;
        case 'clip':
          router.push(
            {
              route: RouteName.Clip,
              routeParams: { slug: payload.data.id },
            },
            {
              ...baseParams,
              [LOOP_PARAM]: payload.data.options?.loop?.toString() ?? LOOP_ON,
            },
          );
          break;
      }
      return;
    }

    if (payload.command === 'stop') {
      router
        .push({
          route: RouteName.Homepage,
        })
        .then(() => {
          sendEvent('stopped');
        });
      return;
    }

    if (!playerController) {
      return;
    }

    // player-based commands
    switch (payload.command) {
      case 'play':
        playerController.play().then(() => {
          sendEvent('played');
        });
        break;
      case 'pause':
        playerController.pause();
        sendEvent('paused');
        break;
      case 'getMuted':
        sendEvent('mutedStatus', playerController.isMuted());
        break;
      case 'setMuted':
        playerController.setMuted(payload.data);
        sendEvent('mutedSet', playerController.isMuted());
        break;
      case 'getPosition':
        sendEvent('positionStatus', playerController.getPosition());
        break;
      case 'setPosition':
        playerController.seekTo(payload.data);
        sendEvent('positionSet', playerController.getPosition());
        break;
      case 'getVolume':
        sendEvent('volumeStatus', playerController.getVolume());
        break;
      case 'setVolume':
        playerController.setVolume(payload.data);
        sendEvent('volumeSet', playerController.getVolume());
        break;
      default:
        const exhausted: never = payload.command;
        return exhausted;
    }
  };
