import { graphql } from 'react-relay/hooks';
import type { TachyonPage } from 'tachyon-next-types';
import type {
  ChannelInitialProps,
  ChannelPathParameters,
} from 'tachyon-page-utils';
import {
  channelIsNotFoundServerside,
  channelPathGetInitialProps,
  channelWatchPageviewTracking,
  getPlayingChannel,
} from 'tachyon-page-utils';
import { MainVideoContainer } from '../../common';
import { useControllerContext } from '../../controller';
import { StreamPlayerManager } from '../../player';
import type { Channel_QueryResponse } from './__generated__/Channel_Query.graphql';

export type ChannelProps = Channel_QueryResponse & ChannelInitialProps;

// istanbul ignore next: trivial
export const Channel: TachyonPage<
  ChannelInitialProps,
  ChannelProps,
  ChannelPathParameters
> = ({ channel }) => {
  const { onPlayerEvent } = useControllerContext();
  const playingChannel = getPlayingChannel(channel);

  if (!playingChannel) {
    // Treat a GraphQL error as a video error since it should be retryable:
    // 1) It's the consumer's responsibility to only request content that exists.
    // 2) We can't distinguish between 404 / 5xx errors in GQL. Assume it's the
    //    latter since the consumer should only be requesting valid content.
    onPlayerEvent('videoErrored');
    return null;
  }

  return (
    <MainVideoContainer>
      <StreamPlayerManager channel={playingChannel} />
    </MainVideoContainer>
  );
};

Channel.displayName = 'Channel';
Channel.getInitialProps = channelPathGetInitialProps;
Channel.isNotFoundServerside = channelIsNotFoundServerside;
Channel.pageviewTracking = channelWatchPageviewTracking;

// eslint-disable-next-line no-unused-expressions
graphql`
  fragment Channel_channel on User {
    ...StreamPlayerManager_channel
    id
    login
    stream {
      id
      game {
        name
      }
    }
  }
`;

Channel.query = graphql`
  query Channel_Query($login: String!) {
    channel: user(login: $login) {
      ...Channel_channel @relay(mask: false)
      hosting {
        ...Channel_channel @relay(mask: false)
      }
    }
  }
`;
