import { graphql } from 'react-relay/hooks';
import type { TachyonPage } from 'tachyon-next-types';
import { defaultPageviewTracking } from 'tachyon-page-utils';
import { RouteName, TachyonLink } from '../../../routing';
import type { Homepage_QueryResponse } from './__generated__/Homepage_Query.graphql';

const CurrentTopChannelsStyles = () => (
  <style>{`
    ul {
      display: grid;
      grid-template-columns: repeat(auto-fit, minmax(400px, 1fr));
      list-style-type: none;
      justify-items: center;
      margin: 0;
      max-height: calc(100vh - 80px);
      padding: 0;
      overflow-y: scroll;
    }

    h1 {
      font-size: 40px;
      height: 40px;
      margin: 0;
      padding: 20px;
    }

    a, a:visited {
      color: lightgray;
      display: block;
      padding: 10px;
      text-decoration: none;
      width: 400px;
    }

    p {
      margin: 0;
      overflow: hidden;
      padding: 5px;
      text-overflow: ellipsis;
      white-space: nowrap;
      width: 100%;
    }

    img {
      width: 100%;
    }
  `}</style>
);

CurrentTopChannelsStyles.displayName = 'CurrentTopChannelsStyles';

export type HomepageProps = Homepage_QueryResponse;

// istanbul ignore next: not used in production
export const Homepage: TachyonPage<{}, HomepageProps> = ({ streams }) => {
  if (process.env.NODE_ENV === 'production') {
    return null;
  }

  // We only render this stream grid during local dev for ease of finding
  // online streams for testing. In production, this is a black 404 page.
  return (
    <>
      <CurrentTopChannelsStyles />
      <main>
        <h1>Current Top Channels</h1>
        <ul>
          {streams?.edges?.map((edge) => {
            if (!edge?.node?.broadcaster) {
              return null;
            }

            const { broadcaster, game, id, previewImageURL } = edge.node;
            const { broadcastSettings, login } = broadcaster;
            const gameName = game?.name;

            if (!login || !broadcastSettings) {
              return null;
            }

            return (
              <li key={id}>
                <TachyonLink route={RouteName.Channel} routeParams={{ login }}>
                  <img src={previewImageURL ?? ''} />
                  <p>{broadcastSettings.title}</p>
                  <p>{gameName ? `${login} - ${gameName}` : login}</p>
                </TachyonLink>
              </li>
            );
          })}
        </ul>
      </main>
    </>
  );
};

Homepage.displayName = 'Homepage';
Homepage.pageviewTracking = defaultPageviewTracking;

Homepage.query =
  process.env.NODE_ENV === 'production'
    ? undefined
    : graphql`
        query Homepage_Query {
          streams(first: 30) {
            edges {
              node {
                id
                previewImageURL(height: 270, width: 480)
                game {
                  name
                }
                broadcaster {
                  broadcastSettings {
                    title
                  }
                  login
                }
              }
            }
          }
        }
      `;
