import type { FC } from 'react';
import type { Variables as RelayQueryVariables } from 'react-relay/hooks';
import { useLazyLoadQuery } from 'react-relay/hooks';
import { Pageview } from 'tachyon-event-tracker';
import { LatencyTransitionComplete } from 'tachyon-latency-tracker';
import type {
  TachyonPageInitialProps,
  TachyonPageProps,
  TachyonPageType,
} from 'tachyon-next-types';
import { DangerousServerSuspense, isValidQueryResponse } from 'tachyon-relay';

type PageBundleProps = {
  Page: TachyonPageType<TachyonPageInitialProps, TachyonPageProps>;
  pageProps: TachyonPageProps;
};

/**
 * Renders a Page complete with proper transition and page reporting
 */
// istanbul ignore next: trivial
const PageBundle: FC<PageBundleProps> = ({ Page, pageProps }) => {
  return (
    <>
      <LatencyTransitionComplete />
      <Pageview {...Page.pageviewTracking(pageProps)} />
      <Page {...pageProps} />
    </>
  );
};
PageBundle.displayName = 'PageBundle';

type PageWithQueryProps = PageBundleProps & {
  relayQueryVariables: RelayQueryVariables;
};

/**
 * Gets query data for pages that need them. On the server, this component only
 * gets data from the store (which is pre-populated from
 * getServersideInitialProps) to prevent wayward requests. On the client,
 * this will fire a GQL request and Suspend if necessary while awaiting its
 * return. If the client-side fetch fails, this will throw an error to be caught
 * by the nearest Error Boundary.
 */
// istanbul ignore next: trivial
const PageWithQuery: FC<PageWithQueryProps> = ({
  Page,
  pageProps,
  relayQueryVariables,
}) => {
  // this component is only rendered when Page.query is non-null
  // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
  const data = useLazyLoadQuery(Page.query!, relayQueryVariables, {
    fetchPolicy:
      typeof window === 'undefined' ? 'store-only' : 'store-and-network',
  });

  if (!isValidQueryResponse(data)) {
    return null;
  }

  return <PageBundle Page={Page} pageProps={{ ...pageProps, ...data }} />;
};
PageWithQuery.displayName = 'PageWithQuery';

type PageRendererProps = PageWithQueryProps;

/**
 * Renders a Page component. If that component requires GQL data, this will also
 * mediate that fetch request.
 */
// istanbul ignore next: trivial
export const PageRenderer: FC<PageRendererProps> = (props) =>
  props.Page.query ? (
    <DangerousServerSuspense fallback={null}>
      <PageWithQuery {...props} />
    </DangerousServerSuspense>
  ) : (
    <PageBundle {...props} />
  );

PageRenderer.displayName = 'PageRenderer';
