import { configureClientLogging, logger } from 'tachyon-logger';
import type { ConfigureTachyonRelayOpts } from 'tachyon-relay';
import { configureTachyonRelay } from 'tachyon-relay';
import {
  getCurrentTwitchDomain,
  isBrowser,
  setCookieDomain,
} from 'tachyon-utils';
import {
  CLIENT_ID,
  FAIL_SERVICES,
  GQL_DEBUG_MODE,
  GQL_SANDBOX_ENDPOINT,
  PLATFORM,
} from '../../../../config';
import { sendEvent } from '../../../controller';
import { videoTagIsPresent } from './videoTagIsPresent';

export function onUnhandledError(): void {
  /*
   * In production, Valence pages have a video element, or are the 404 or Error
   * page.
   *
   * If we receive a top level unhandled error, we check to see if it was fatal
   * by checking to see if the video player is present.
   */
  const fatal = !videoTagIsPresent();
  if (fatal) {
    sendEvent('appErrored');
  }
}

// istanbul ignore next: cost outweighs value
export function configure(): void {
  const relayOpts: ConfigureTachyonRelayOpts = {
    clientId: CLIENT_ID,
    errorsArrayIsFatal: () => false,
  };

  if (process.env.NODE_ENV !== 'production' && GQL_DEBUG_MODE) {
    relayOpts.debug = GQL_DEBUG_MODE;
    relayOpts.gqlEndpoint = GQL_SANDBOX_ENDPOINT;
  }

  if (isBrowser()) {
    const buildId = process.env.BUILD_ID;
    // reach into next data since we're outside of the react tree on the client
    const { appEnvironment, language } = window.__NEXT_DATA__.props;

    logger.debug({
      category: 'configure',
      context: {
        // eslint-disable-next-line @typescript-eslint/no-unsafe-assignment
        appEnvironment,
        buildId,
        // eslint-disable-next-line @typescript-eslint/no-unsafe-assignment
        language,
      },
      message: 'Configuring application',
      package: 'valence',
    });

    if (process.env.NODE_ENV === 'production') {
      // configure error reporting (this should happen first/asap)
      try {
        configureClientLogging({
          // eslint-disable-next-line @typescript-eslint/no-unsafe-assignment
          appEnvironment,
          appSentryDsn: process.env.SENTRY_DSN,
          buildId,
          // eslint-disable-next-line @typescript-eslint/no-unsafe-assignment
          language,
          onUnhandledError,
          platform: PLATFORM,
        });
      } catch {
        // If this fails still try to start the app, don't bother logging though since our
        // logger isn't shipping logs.
      }
    }

    // configure cookie domain
    try {
      const domain = getCurrentTwitchDomain(window.location.hostname);
      if (domain) {
        setCookieDomain(domain);
        // TODO: audit necessity
        // https://jira.twitch.com/browse/MWC-2473
        document.domain = domain;
        logger.debug({
          category: 'configure',
          message: `Setting the domain for cookies and the document to ${domain}`,
          package: 'valence',
        });
      }
    } catch {
      // This can throw a security exception if the page isn't on a twitch.tv
      // subdomain. If it does then there isn't really much we can do about
      // this. If we didn't have staging domains off of twitch.tv then we
      // could consider this an _actual_ security error.
      logger.warn({
        category: 'configure',
        message: `Failed to set domain for ${window.location.hostname}`,
        package: 'valence',
      });
    }

    // by default we only simulate failure in the browser to enable a better
    // debugging experience
    if (
      process.env.NODE_ENV !== 'production' &&
      GQL_DEBUG_MODE &&
      FAIL_SERVICES.length
    ) {
      relayOpts.failServices = FAIL_SERVICES;
    }
  }

  configureTachyonRelay(relayOpts);
}
