import type { InitEnvironmentOpts, TwitchPayloadError } from 'tachyon-relay';
import { HTTPStatusCode } from 'tachyon-type-library';
import { buildCacheHeader } from 'tachyon-utils';
import { DEFAULT_CACHE_VALUES } from '../../../../config';
import { getServersideInitialProps } from '.';

let mockFetchQueryValue: Promise<any> | undefined = undefined;
jest.mock('react-relay/hooks', () => ({
  ...jest.requireActual('react-relay/hooks'),
  fetchQuery: jest.fn(() => ({ toPromise: () => mockFetchQueryValue })),
}));

let mockNonFatalErrors = false;
jest.mock('tachyon-relay', () => ({
  ...jest.requireActual('tachyon-relay'),
  initEnvironment: jest.fn((opts: InitEnvironmentOpts) => {
    if (mockNonFatalErrors) {
      opts.fetchQueryOpts!.nonFatalErrorsHandler!([
        'uh oh',
      ] as unknown as TwitchPayloadError[]);
    }
    return jest.requireActual('tachyon-relay').initEnvironment(opts);
  }),
}));

const mockTachyonRes = { setHeader: jest.fn() };

beforeEach(() => {
  mockFetchQueryValue = undefined;
  mockTachyonRes.setHeader.mockReset();
});

describe(getServersideInitialProps, () => {
  describe('serversideError', () => {
    it('true when tachyonCtx.err', async () => {
      const res = await getServersideInitialProps({
        Component: {} as any,
        fetchQueryOpts: {} as any,
        relayQueryVariables: {} as any,
        tachyonCtx: {
          err: new Error('oh no'),
          res: { ...mockTachyonRes, statusCode: 200 },
        } as any,
      });

      expect(res.serversideError).toEqual(true);
    });

    it('true when 500 status code', async () => {
      const res = await getServersideInitialProps({
        Component: {} as any,
        fetchQueryOpts: {} as any,
        relayQueryVariables: {} as any,
        tachyonCtx: {
          res: { ...mockTachyonRes, statusCode: 500 },
        } as any,
      });

      expect(res.serversideError).toEqual(true);
    });

    it('false otherwise', async () => {
      const res = await getServersideInitialProps({
        Component: {} as any,
        fetchQueryOpts: {} as any,
        relayQueryVariables: {} as any,
        tachyonCtx: {
          res: { ...mockTachyonRes, statusCode: 200 },
        } as any,
      });

      expect(res.serversideError).toEqual(false);
    });
  });

  describe('isGQLPartialSuccess', () => {
    describe('Component with query', () => {
      it('true when partial request failures', async () => {
        mockNonFatalErrors = true;

        const res = await getServersideInitialProps({
          Component: { query: true } as any,
          fetchQueryOpts: {} as any,
          relayQueryVariables: {} as any,
          tachyonCtx: {
            res: { ...mockTachyonRes, statusCode: undefined },
          } as any,
        });

        expect(res.isGQLPartialSuccess).toEqual(true);
        mockNonFatalErrors = false;
      });

      it('false otherwise', async () => {
        const res = await getServersideInitialProps({
          Component: { query: true } as any,
          fetchQueryOpts: {} as any,
          relayQueryVariables: {} as any,
          tachyonCtx: {
            res: { ...mockTachyonRes, statusCode: undefined },
          } as any,
        });

        expect(res.isGQLPartialSuccess).toEqual(false);
      });
    });
  });

  describe('relayQueryRecords', () => {
    describe('Component with query', () => {
      it('returns relay query records', async () => {
        mockFetchQueryValue = Promise.resolve(true);

        const res = await getServersideInitialProps({
          Component: { query: true } as any,
          fetchQueryOpts: {} as any,
          relayQueryVariables: {} as any,
          tachyonCtx: {
            res: { ...mockTachyonRes, statusCode: 200 },
          } as any,
        });

        expect(res.relayQueryRecords).not.toBeUndefined();
      });
    });

    describe('Component without query', () => {
      it('does not return relay query records', async () => {
        const res = await getServersideInitialProps({
          Component: {} as any,
          fetchQueryOpts: {} as any,
          relayQueryVariables: {} as any,
          tachyonCtx: {
            res: { ...mockTachyonRes, statusCode: 200 },
          } as any,
        });

        expect(res.relayQueryRecords).toBeUndefined();
      });
    });
  });

  describe('headers', () => {
    const defaultCacheValues = buildCacheHeader(DEFAULT_CACHE_VALUES);

    describe('Component with query', () => {
      it('request failures set the status code to BadGateway', async () => {
        const error = new Error('oh no');
        mockFetchQueryValue = Promise.reject(error);
        const res = { ...mockTachyonRes, statusCode: undefined };

        await getServersideInitialProps({
          Component: { query: true } as any,
          fetchQueryOpts: {} as any,
          relayQueryVariables: {} as any,
          tachyonCtx: { res } as any,
        });

        expect(res.statusCode).toEqual(HTTPStatusCode.BadGateway);
      });

      it('partial request failures set the status code to BadGateway', async () => {
        mockNonFatalErrors = true;
        const res = { ...mockTachyonRes, statusCode: undefined };

        await getServersideInitialProps({
          Component: { query: true } as any,
          fetchQueryOpts: {} as any,
          relayQueryVariables: {} as any,
          tachyonCtx: { res } as any,
        });

        expect(res.statusCode).toEqual(HTTPStatusCode.BadGateway);
        mockNonFatalErrors = false;
      });

      describe('Component.isNotFoundServerside', () => {
        it('sets the status code to NotFound', async () => {
          mockFetchQueryValue = Promise.resolve(true);
          const res = { ...mockTachyonRes, statusCode: undefined };

          await getServersideInitialProps({
            Component: { isNotFoundServerside: () => true, query: true } as any,
            fetchQueryOpts: {} as any,
            relayQueryVariables: {} as any,
            tachyonCtx: { res } as any,
          });

          expect(res.statusCode).toEqual(HTTPStatusCode.NotFound);
        });
      });

      it('sets "Cache-Control" header when valid GQL response', async () => {
        mockFetchQueryValue = Promise.resolve(true);

        await getServersideInitialProps({
          Component: { query: true } as any,
          fetchQueryOpts: {} as any,
          relayQueryVariables: {} as any,
          tachyonCtx: { res: mockTachyonRes } as any,
        });

        expect(mockTachyonRes.setHeader).toHaveBeenCalledWith(
          'Cache-Control',
          defaultCacheValues,
        );
      });

      it('does not set "Cache-Control" header when GQL request errors', async () => {
        mockFetchQueryValue = Promise.resolve(null);

        await getServersideInitialProps({
          Component: { query: true } as any,
          fetchQueryOpts: {} as any,
          relayQueryVariables: {} as any,
          tachyonCtx: { res: mockTachyonRes } as any,
        });

        expect(mockTachyonRes.setHeader).toHaveBeenCalledTimes(0);
      });
    });

    describe('Component without query', () => {
      it('sets "Cache-Control" header', async () => {
        await getServersideInitialProps({
          Component: {} as any,
          fetchQueryOpts: {} as any,
          relayQueryVariables: {} as any,
          tachyonCtx: { res: mockTachyonRes } as any,
        });

        expect(mockTachyonRes.setHeader).toHaveBeenCalledWith(
          'Cache-Control',
          defaultCacheValues,
        );
      });
    });
  });
});
