import App from 'next/app';
import { PlayerControllerRoot } from 'pulsar';
import { useState } from 'react';
import type { Variables as RelayQueryVariables } from 'react-relay/hooks';
import type { DynamicSettings } from 'tachyon-dynamic-settings';
import { DynamicSettingsRoot } from 'tachyon-dynamic-settings';
import type { AppEnvironment } from 'tachyon-environment';
import { EnvironmentRoot } from 'tachyon-environment';
import { Pageview } from 'tachyon-event-tracker';
import type { ClientSideErrorBoundaryProps } from 'tachyon-more-ui';
import { ClientSideErrorBoundary } from 'tachyon-more-ui';
import { RouterUtilsRoot } from 'tachyon-next-routing-utils';
import type {
  TachyonAppBaseInitialProps,
  TachyonAppBaseProps,
  TachyonAppFC,
  TachyonPageInitialProps,
} from 'tachyon-next-types';
import { isBrowser } from 'tachyon-utils';
import type { ValenceDynamicSettings } from '../../../config';
import {
  AUTOPLAY_PARAM,
  CLIENT_APP,
  INITIAL_MUTED_STATE_PARAM,
  INITIAL_VOLUME_PARAM,
  LOOP_PARAM,
  PLATFORM,
  START_FROM_TIME_PARAM,
} from '../../../config';
import {
  ControllerRoot,
  isUnauthorizedAccess,
  sendEvent,
} from '../../controller';
import { PageErrorBoundary } from './PageErrorBoundary';
import { PageRenderer } from './PageRenderer';
import { configure } from './configure';
import {
  ValenceEventReporterRoot,
  ValenceEventTrackerRoot,
  ValenceLatencyTrackerRoot,
  ValenceRelayEnvironmentRoot,
} from './context-root-adapters';
import type { ValenceServersideInitialProps } from './getServersideInitialProps';
import {
  getServersideInitialProps,
  isServersideContext,
} from './getServersideInitialProps';

configure();

// list of query params to protect from being stripped by the query-param
// handling in RouterUtilsRoot
const PRESERVED_QUERY_PARAMS = [
  // controls initial autoPlay state of all video mediums
  AUTOPLAY_PARAM,
  // controls initial muted state of all video mediums
  INITIAL_MUTED_STATE_PARAM,
  // controls initial volume for all video mediums
  INITIAL_VOLUME_PARAM,
  // controls whether or not to loop for clips
  LOOP_PARAM,
  // controls video start position for VOD
  START_FROM_TIME_PARAM,
];

/**
 * PersistedValenceAppInitialProps is comprised of values that we need to transport
 * from the server to client, and we lean on next's `getInitialProps`
 * serialization functionality to achieve that. The values are set on the
 * server, consumed from __NEXT_DATA__ via normal next functionality for the
 * first page render, and then manually re-consumed on subsequent app re-renders
 * to ensure their continued availability.
 */
type ValencePersistedAppInitialProps = {
  appEnvironment: AppEnvironment;
  dynamicSettings: DynamicSettings<ValenceDynamicSettings>;
};

// Be very conservative about what is put in here, since all of this will get
// serialized for being sent to the client and big/complex objects both don't
// do very well with this and also cause a ton of page bloat.

type ValenceAppInitialProps = Partial<ValenceServersideInitialProps> &
  TachyonAppBaseInitialProps &
  ValencePersistedAppInitialProps & {
    currentAsPath: string;
    relayQueryVariables: RelayQueryVariables;
  };

export type ValenceAppProps = TachyonAppBaseProps & ValenceAppInitialProps;

export const ValenceApp: TachyonAppFC<ValenceAppInitialProps, ValenceAppProps> =
  ({
    Component,
    appEnvironment,
    currentAsPath,
    dynamicSettings,
    pageProps,
    relayQueryRecords,
    relayQueryVariables,
    serversideError,
  }) => {
    const [isInError, setIsInError] = useState(false);
    function onError() {
      setIsInError(true);
    }

    const isUnauthorized = isUnauthorizedAccess(appEnvironment);

    const commonErrorBoundaryProps: Pick<
      ClientSideErrorBoundaryProps,
      'app' | 'currentPath' | 'errorStateComponent' | 'isDevelopment'
    > = {
      app: 'valence',
      currentPath: currentAsPath,
      errorStateComponent: () => null,
      isDevelopment: process.env.NODE_ENV === 'development',
    };

    return (
      <ClientSideErrorBoundary
        {...commonErrorBoundaryProps}
        boundaryName="ValenceRootBoundary"
        isRootBoundary
        onError={() => {
          // ControllerRoot isn't mounted, send the event directly
          sendEvent('appErrored');
        }}
      >
        <RouterUtilsRoot preservedParams={PRESERVED_QUERY_PARAMS}>
          <DynamicSettingsRoot settings={dynamicSettings}>
            <EnvironmentRoot
              common={{
                appEnvironment,
                appVersion: process.env.BUILD_ID,
                clientApp: CLIENT_APP,
                language: 'en',
                platform: PLATFORM,
              }}
              networkStatusDebounceWait={null}
            >
              <ValenceEventReporterRoot>
                <ValenceEventTrackerRoot
                  inError={serversideError ?? isInError}
                  isUnauthorizedAccess={isUnauthorized}
                >
                  <ValenceRelayEnvironmentRoot records={relayQueryRecords}>
                    <ValenceLatencyTrackerRoot>
                      <PlayerControllerRoot>
                        <ControllerRoot isUnauthorizedAccess={isUnauthorized}>
                          <PageErrorBoundary
                            {...commonErrorBoundaryProps}
                            boundaryName="ValenceAppBoundary"
                            onError={onError}
                          >
                            {isUnauthorized ? (
                              <Pageview />
                            ) : serversideError ? null : (
                              <PageRenderer
                                Page={Component}
                                pageProps={pageProps}
                                relayQueryVariables={relayQueryVariables}
                              />
                            )}
                          </PageErrorBoundary>
                        </ControllerRoot>
                      </PlayerControllerRoot>
                    </ValenceLatencyTrackerRoot>
                  </ValenceRelayEnvironmentRoot>
                </ValenceEventTrackerRoot>
              </ValenceEventReporterRoot>
            </EnvironmentRoot>
          </DynamicSettingsRoot>
        </RouterUtilsRoot>
      </ClientSideErrorBoundary>
    );
  };

ValenceApp.displayName = 'ValenceApp';

ValenceApp.getInitialProps = async (appContext) => {
  const { Component, ctx: tachyonCtx } = appContext;

  const persistedProps: ValencePersistedAppInitialProps = isBrowser()
    ? {
        /* eslint-disable @typescript-eslint/no-unsafe-assignment */
        appEnvironment: window.__NEXT_DATA__.props.appEnvironment,
        dynamicSettings: window.__NEXT_DATA__.props.dynamicSettings,
        /* eslint-enable @typescript-eslint/no-unsafe-assignment */
      }
    : {
        /* eslint-disable @typescript-eslint/no-non-null-assertion */
        appEnvironment: tachyonCtx.req!.tachyon.appEnvironment,
        dynamicSettings: tachyonCtx.req!.tachyon.dynamicSettings,
        /* eslint-enable @typescript-eslint/no-non-null-assertion */
      };

  const initialProps = await App.getInitialProps(appContext);

  const relayQueryVariables: RelayQueryVariables =
    (initialProps.pageProps as TachyonPageInitialProps).queryVariables ?? {};

  const serverSideInitialProps =
    // typeof window and tachyonCtx.req/res should move in sync, here just for types
    typeof window === 'undefined' && isServersideContext(tachyonCtx)
      ? await getServersideInitialProps({
          Component,
          fetchQueryOpts: {},
          relayQueryVariables,
          tachyonCtx,
        })
      : undefined;

  return {
    ...initialProps,
    ...persistedProps,
    ...serverSideInitialProps,
    currentAsPath: tachyonCtx.asPath!, // eslint-disable-line @typescript-eslint/no-non-null-assertion
    relayQueryVariables,
  };
};
