import { lorem } from 'faker';
import { RouterUtilsRoot } from 'tachyon-next-routing-utils';
import type {
  TachyonPage,
  TachyonPageInitialProps,
  TachyonPageProps,
} from 'tachyon-next-types';
import { defaultPageviewTracking } from 'tachyon-page-utils';
import { createMountWrapperFactory } from 'tachyon-test-utils';
import { isUnauthorizedAccess, sendEvent } from '../../controller';
import { PageRenderer } from './PageRenderer';
import { getServersideInitialProps } from './getServersideInitialProps';
import type { ValenceAppProps } from '.';
import { ValenceApp } from '.';

jest.mock('tachyon-relay', () => ({
  ...jest.requireActual('tachyon-relay'),
  RequestInfoRoot: (props: any) => props.children,
}));

jest.mock('tachyon-event-tracker', () => ({
  ...jest.requireActual('tachyon-event-tracker'),
  Pageview: jest.fn(() => null),
}));

const mockSendEvent = sendEvent as jest.Mock;
const mockIsUnauthorizedAccess = isUnauthorizedAccess as jest.Mock;
jest.mock('../../controller', () => ({
  ControllerRoot: (props: any) => props.children,
  isUnauthorizedAccess: jest.fn(),
  sendEvent: jest.fn(),
  useControllerContext: () => ({
    onPlayerEvent: jest.fn(),
  }),
}));

/* supress console messages when running tests */
jest.mock('tachyon-logger', () => ({ logger: { debug: jest.fn() } }));
const mockConsoleError = jest.spyOn(console, 'error').mockImplementation();
afterAll(() => {
  mockConsoleError.mockRestore();
});

const mockRouterUtilsRoot = RouterUtilsRoot as jest.Mock;
jest.mock('tachyon-next-routing-utils', () => {
  return {
    ...jest.requireActual('tachyon-next-routing-utils'),
    RouterUtilsRoot: jest.fn((props: any) => props.children),
  };
});

jest.mock('./configure', () => ({
  configure: jest.fn(),
}));

jest.mock('next/app', () => {
  const app = () => false;
  app.getInitialProps = () => ({ pageProps: { queryVariables: {} } });
  return {
    __esModule: true,
    default: app,
  };
});

jest.mock('tachyon-debug-reporter', () => ({
  configTachyonDebug: jest.fn(),
}));

jest.mock('tachyon-environment', () => ({
  ...jest.requireActual('tachyon-environment'),
  EnvironmentRoot: (props: any) => props.children,
}));

jest.mock('./PageRenderer', () => ({
  ...jest.requireActual('./PageRenderer'),
  PageRenderer: jest.fn(({ Page }) => <Page />),
}));
const mockPageRenderer = PageRenderer as jest.Mock;

jest.mock('./getServersideInitialProps', () => ({
  ...jest.requireActual('./getServersideInitialProps'),
  getServersideInitialProps: jest.fn(() => ({ isLoggedInServerside: false })),
}));
const mockGetServersideInitialProps = getServersideInitialProps as jest.Mock;

const mockPage = (
  renderer: any = () => <div />,
): TachyonPage<TachyonPageInitialProps, TachyonPageProps> => {
  const page = renderer;
  page.pageviewTracking = defaultPageviewTracking;

  return page;
};

describe(ValenceApp, () => {
  const pageProps = { foo: 'bar' };

  const setup = createMountWrapperFactory(
    ValenceApp as any,
    (): ValenceAppProps => ({
      Component: mockPage(),
      appEnvironment: 'development',
      currentAsPath: '/',
      dynamicSettings: {
        environment: 'development',
        experiments: {},
        flumeAllowedEvents: [],
        flumeAllowedProperties: [],
        spadeUrl: '//spade',
      },
      pageProps: pageProps as any,
      relayQueryVariables: {},
      router: {} as any,
    }),
  );

  it('empty renders when there is an SSR error', () => {
    const { wrapper } = setup({ serversideError: true });

    expect(wrapper).toBeEmptyRender();
  });

  it('empty renders when unauthorized', () => {
    mockIsUnauthorizedAccess.mockImplementationOnce(() => true);
    const { wrapper } = setup();

    expect(wrapper).toBeEmptyRender();
  });

  it('renders the ValenceAppBoundary when the <Page/> fails', () => {
    const { wrapper } = setup({
      Component: mockPage(() => {
        throw new Error('oh no');
      }),
    });

    expect(wrapper.find({ boundaryName: 'ValenceAppBoundary' })).toExist();
  });

  it('renders the ValenceRootBoundary when the app fails', () => {
    mockRouterUtilsRoot.mockImplementationOnce(() => {
      throw new Error('oh no');
    });
    const { wrapper } = setup();

    expect(wrapper.find({ boundaryName: 'ValenceRootBoundary' })).toExist();
    expect(mockSendEvent).toHaveBeenCalledTimes(1);
    expect(mockSendEvent).toHaveBeenCalledWith('appErrored');
  });

  it('renders the <Page /> when there is not an SSR error', () => {
    const { props, wrapper } = setup();

    expect(wrapper.find(props.Component as any)).toExist();
  });

  it('renders the ValenceAppBoundary when the PageRenderer has an error', () => {
    mockPageRenderer.mockImplementationOnce(() => {
      throw new Error('oh no');
    });
    const { props, wrapper } = setup();

    expect(wrapper.find({ boundaryName: 'ValenceAppBoundary' })).toExist();
    expect(wrapper.find(props.Component as any)).not.toExist();
  });
});

describe('ValenceApp.getInitialProps', () => {
  describe('client side', () => {
    const mockNextData = {
      appEnvironment: 'staging',
      dynamicSettings: { spadeUrl: lorem.word() },
    };

    beforeEach(() => {
      global.window.__NEXT_DATA__ = { props: mockNextData } as any;
    });

    // @ts-expect-error: tests
    afterEach(() => delete global.window.__NEXT_DATA__);

    it('extracts __NEXT_DATA__ fields', async () => {
      const ip = await ValenceApp.getInitialProps({ ctx: {} } as any);

      expect(ip).toEqual(expect.objectContaining(mockNextData));
    });
  });

  describe('server side', () => {
    let mockWindow: any;
    beforeEach(() => {
      mockWindow = jest.spyOn(global as any, 'window', 'get');
      mockWindow.mockImplementation(() => undefined);
    });
    afterEach(() => {
      mockWindow.mockRestore();
    });

    const mockReqData = {
      appEnvironment: 'staging',
      dynamicSettings: { spadeUrl: lorem.word() },
    };

    const deviceId = lorem.word();
    const mockCtx = {
      ctx: { req: { tachyon: { ...mockReqData, deviceId } }, res: {} },
    };

    it('extracts req fields when', async () => {
      const ip = await ValenceApp.getInitialProps(mockCtx as any);

      expect(ip).toEqual(expect.objectContaining(mockReqData));
    });

    it('invokes getServersideInitialProps', async () => {
      await ValenceApp.getInitialProps(mockCtx as any);

      expect(mockGetServersideInitialProps).toHaveBeenCalledTimes(1);
    });
  });
});
