import type { ParsedUrlQuery } from 'querystring';
import type { PulsarCoreProps } from 'pulsar';
import {
  PulsarCore,
  usePlaybackAutoplayStatus,
  usePlaybackState,
  usePlayerError,
} from 'pulsar';
import type { FC } from 'react';
import { useEffect, useRef } from 'react';
import { useStaticEnvironment } from 'tachyon-environment';
import { useCustomLatency } from 'tachyon-latency-tracker';
import { useRouterUtils } from 'tachyon-next-routing-utils';
import { flattenHeaderOrParam, useConstCallback } from 'tachyon-utils';
import {
  AUTOPLAY_OFF,
  AUTOPLAY_PARAM,
  CLIENT_ID,
  DEFAULT_INITIAL_VOLUME,
  INITIAL_MUTED_STATE_PARAM,
  INITIAL_VOLUME_PARAM,
  START_UNMUTED,
} from '../../../config';
import { useControllerContext } from '../../controller';

export type BasePlayerProps = Pick<
  PulsarCoreProps,
  'loop' | 'params' | 'startTime'
>;

function getSoundSettings(
  currentQuery: ParsedUrlQuery,
): Pick<PulsarCoreProps, 'muted' | 'volume'> {
  const muted =
    flattenHeaderOrParam(currentQuery[INITIAL_MUTED_STATE_PARAM]) !==
    START_UNMUTED;

  const customVolume = parseFloat(
    flattenHeaderOrParam(currentQuery[INITIAL_VOLUME_PARAM]) ?? '',
  );

  const volume =
    0 <= customVolume && customVolume <= 1
      ? customVolume
      : parseFloat(DEFAULT_INITIAL_VOLUME);

  return { muted, volume };
}

// istanbul ignore next: trivial
export const BasePlayer: FC<BasePlayerProps> = (props) => {
  const bufferingRef = useRef(false);
  const { client, common } = useStaticEnvironment();
  const { onPlayerEvent } = useControllerContext();
  const { currentQuery } = useRouterUtils();
  const autoPlay =
    flattenHeaderOrParam(currentQuery[AUTOPLAY_PARAM]) !== AUTOPLAY_OFF;

  const state = usePlaybackState();
  const error = usePlayerError();
  const autoPlayFailureStatus = usePlaybackAutoplayStatus();

  const reportLatencyEvent = useCustomLatency();
  const onFirstControlsAppear = useConstCallback(
    reportLatencyEvent('playerReady'),
  );

  const reportPlayerPlaying = useConstCallback(
    reportLatencyEvent('playerPlaying'),
  );

  useEffect(() => {
    switch (state) {
      case 'Playing':
        reportPlayerPlaying();
        if (bufferingRef.current) {
          onPlayerEvent('bufferRefilled');
          bufferingRef.current = false;
        }
        break;
      case 'Buffering':
        onPlayerEvent('bufferEmptied');
        bufferingRef.current = true;
        break;
      case 'Ended':
        onPlayerEvent('contentEnded');
        break;
    }
  }, [state, reportPlayerPlaying, onPlayerEvent]);

  useEffect(() => {
    if (error) {
      onPlayerEvent('videoErrored');
    }
  }, [error, onPlayerEvent]);

  useEffect(() => {
    if (autoPlayFailureStatus === 'autoplay-blocked') {
      onPlayerEvent('autoplayFailed');
    }
  }, [autoPlayFailureStatus, onPlayerEvent]);

  return (
    <PulsarCore
      autoPlay={autoPlay}
      controls={false}
      environment={{
        benchmarkSessionID: client.sessionID,
        clientAPIID: CLIENT_ID,
        clientApp: common.clientApp,
        clientBuildID: common.appVersion,
        deviceID: client.deviceID,
        platform: common.platform,
      }}
      onFirstControlsAppear={onFirstControlsAppear}
      onSeekCompleted={(position) => {
        onPlayerEvent('seekCompleted', position);
      }}
      {...getSoundSettings(currentQuery)}
      {...props}
    />
  );
};

BasePlayer.displayName = 'BasePlayer';
