import { random } from 'faker';
import { PulsarCore } from 'pulsar';
import { createShallowWrapperFactory } from 'tachyon-test-utils';
import {
  AUTOPLAY_OFF,
  AUTOPLAY_ON,
  AUTOPLAY_PARAM,
  DEFAULT_INITIAL_VOLUME,
  INITIAL_MUTED_STATE_PARAM,
  INITIAL_VOLUME_PARAM,
  START_MUTED,
  START_UNMUTED,
} from '../../../config';
import { BasePlayer } from '.';

let mockCurrentQuery = {};
jest.mock('tachyon-next-routing-utils', () => {
  const { mockRouterUtils } = jest.requireActual('tachyon-next-routing-utils');
  return {
    useRouterUtils: () => mockRouterUtils({ currentQuery: mockCurrentQuery }),
  };
});

describe(BasePlayer, () => {
  beforeEach(() => {
    mockCurrentQuery = {};
  });

  const setup = createShallowWrapperFactory(BasePlayer, () => ({
    loop: true,
    params: {} as any,
    reportLatencyEvent: jest.fn(),
  }));

  describe('handles autoPlay', () => {
    it('autoPlay by default', () => {
      const { wrapper } = setup();
      expect(wrapper.find(PulsarCore)).toHaveProp({ autoPlay: true });
    });

    it('autoPlay when query param is present with AUTOPLAY_ON value', () => {
      mockCurrentQuery = { [AUTOPLAY_PARAM]: AUTOPLAY_ON };
      const { wrapper } = setup();
      expect(wrapper.find(PulsarCore)).toHaveProp({ autoPlay: true });
    });

    it('does not autoPlay when query param is present with AUTOPLAY_OFF value', () => {
      mockCurrentQuery = { [AUTOPLAY_PARAM]: AUTOPLAY_OFF };
      const { wrapper } = setup();
      expect(wrapper.find(PulsarCore)).toHaveProp({ autoPlay: false });
    });

    it('use default with unknown AUTOPLAY_PARAM value', () => {
      mockCurrentQuery = { [AUTOPLAY_PARAM]: random.alphaNumeric() };
      const { wrapper } = setup();
      expect(wrapper.find(PulsarCore)).toHaveProp({ autoPlay: true });
    });
  });

  describe('handles autoMute', () => {
    it('mutes by default', () => {
      const { wrapper } = setup();
      expect(wrapper.find(PulsarCore)).toHaveProp({ muted: true });
    });

    it('mutes when query param is present with START_MUTED value', () => {
      mockCurrentQuery = { [INITIAL_MUTED_STATE_PARAM]: START_MUTED };
      const { wrapper } = setup();
      expect(wrapper.find(PulsarCore)).toHaveProp({ muted: true });
    });

    it('does not mute when query param is present with START_UNMUTED value', () => {
      mockCurrentQuery = { [INITIAL_MUTED_STATE_PARAM]: START_UNMUTED };
      const { wrapper } = setup();
      expect(wrapper.find(PulsarCore)).toHaveProp({ muted: false });
    });

    it('use default with unknown INITIAL_MUTED_STATE_PARAM value', () => {
      mockCurrentQuery = {
        [INITIAL_MUTED_STATE_PARAM]: random.alphaNumeric(),
      };
      const { wrapper } = setup();
      expect(wrapper.find(PulsarCore)).toHaveProp({ muted: true });
    });
  });

  describe('sets the initial volume', () => {
    it('sets the volume when INITIAL_VOLUME_PARAM is set', () => {
      mockCurrentQuery = { [INITIAL_VOLUME_PARAM]: '0.7' };
      const { wrapper } = setup();
      expect(wrapper.find(PulsarCore)).toHaveProp({ volume: 0.7 });
    });

    it('falls back to DEFAULT_INITIAL_VOLUME when INITIAL_VOLUME_PARAM is out of bounds', () => {
      // under 0.0
      mockCurrentQuery = { [INITIAL_VOLUME_PARAM]: '1.1' };
      const { wrapper: wrapper1 } = setup();
      expect(wrapper1.find(PulsarCore)).toHaveProp({
        volume: parseFloat(DEFAULT_INITIAL_VOLUME),
      });

      // over 1.0
      mockCurrentQuery = { [INITIAL_VOLUME_PARAM]: '-0.1' };
      const { wrapper: wrapper2 } = setup();
      expect(wrapper2.find(PulsarCore)).toHaveProp({
        volume: parseFloat(DEFAULT_INITIAL_VOLUME),
      });
    });

    it('falls back to DEFAULT_INITIAL_VOLUME when INITIAL_VOLUME_PARAM is missing', () => {
      mockCurrentQuery = {};
      const { wrapper } = setup();
      expect(wrapper.find(PulsarCore)).toHaveProp({
        volume: parseFloat(DEFAULT_INITIAL_VOLUME),
      });
    });
  });
});
