import { PlayerType, useClipAnalytics } from 'pulsar';
import type { FC } from 'react';
import { graphql, useFragment } from 'react-relay/hooks';
import { useStaticEnvironment } from 'tachyon-environment';
import { useRouterUtils } from 'tachyon-next-routing-utils';
import { clipPulsarParams } from 'tachyon-page-utils';
import { flattenHeaderOrParam } from 'tachyon-utils';
import { LOOP_OFF, LOOP_PARAM } from '../../../config';
import type { LoadedEventData } from '../../controller';
import { BasePlayer } from '../BasePlayer';
import { useOnloadMetadataEmitter } from '../useOnLoadMetadataEmitter';
import { useTrackingEvent } from '../useTrackingEvent';
import type {
  ClipPlayerManager_clip,
  ClipPlayerManager_clip$key,
} from './__generated__/ClipPlayerManager_clip.graphql';

const clipPlayerManagerFragment = graphql`
  fragment ClipPlayerManager_clip on Clip {
    id
    slug
    broadcaster {
      id
      displayName
      login
      roles {
        isPartner
      }
      self {
        subscriptionBenefit {
          id
        }
      }
    }
    game {
      id
      name
    }
    durationSeconds
    video {
      createdAt
    }
  }
`;

export type ClipPlayerManagerProps = {
  clip: ClipPlayerManager_clip$key;
};

// istanbul ignore next: trivial
export function clipMetadata(clip: ClipPlayerManager_clip): LoadedEventData {
  return {
    broadcastTimestamp: clip.video?.createdAt ?? null,
    duration: clip.durationSeconds,
    type: 'clip',
  };
}

export const ClipPlayerManager: FC<ClipPlayerManagerProps> = ({
  clip: clipRef,
}) => {
  const clip = useFragment(clipPlayerManagerFragment, clipRef);
  const { currentQuery } = useRouterUtils();
  const loop = flattenHeaderOrParam(currentQuery[LOOP_PARAM]) !== LOOP_OFF;

  useOnloadMetadataEmitter(clipMetadata(clip));

  const {
    client: { sessionID },
  } = useStaticEnvironment();

  const onTrackingEvent = useTrackingEvent();

  useClipAnalytics({
    clip,
    currentUser: null,
    onTrackingEvent,
    tracking: {
      playerType: PlayerType.Pulsar,
      sessionId: sessionID,
    },
  });

  return <BasePlayer loop={loop} params={clipPulsarParams({ clip })} />;
};

ClipPlayerManager.displayName = 'ClipPlayerManager';
