import { random } from 'faker';
import { useFragment } from 'react-relay/hooks';
import { clipPulsarParams } from 'tachyon-page-utils';
import { createShallowWrapperFactory } from 'tachyon-test-utils';
import { LOOP_OFF, LOOP_ON, LOOP_PARAM } from '../../../config';
import { BasePlayer } from '../BasePlayer';
import { mockClip } from './test-mocks';
import { ClipPlayerManager } from '.';

jest.mock('react-relay/hooks', () => ({
  ...jest.requireActual('react-relay/hooks'),
  useFragment: jest.fn(),
}));
const mockUseFragment = useFragment as jest.Mock;

let mockCurrentQuery = {};
jest.mock('tachyon-next-routing-utils', () => {
  const { mockRouterUtils } = jest.requireActual('tachyon-next-routing-utils');
  return {
    useRouterUtils: () => mockRouterUtils({ currentQuery: mockCurrentQuery }),
  };
});

describe(ClipPlayerManager, () => {
  beforeEach(() => {
    mockUseFragment.mockReset();
    mockCurrentQuery = {};
  });

  const setup = createShallowWrapperFactory(ClipPlayerManager, () => ({
    clip: {} as any,
  }));

  it('renders a player', () => {
    mockUseFragment.mockImplementationOnce(() => mockClip);
    const { wrapper } = setup();
    expect(wrapper.find(BasePlayer)).toHaveProp({
      params: clipPulsarParams({ clip: mockClip }),
    });
  });

  describe('handles looping', () => {
    it('loops by default', () => {
      mockUseFragment.mockImplementationOnce(() => mockClip);
      const { wrapper } = setup();
      expect(wrapper.find(BasePlayer)).toHaveProp({ loop: true });
    });

    it('loops when query param is present with LOOP_ON value', () => {
      mockUseFragment.mockImplementationOnce(() => mockClip);
      mockCurrentQuery = { [LOOP_PARAM]: LOOP_ON };
      const { wrapper } = setup();
      expect(wrapper.find(BasePlayer)).toHaveProp({ loop: true });
    });

    it('does not loop when query param is present with LOOP_OFF value', () => {
      mockUseFragment.mockImplementationOnce(() => mockClip);
      mockCurrentQuery = { [LOOP_PARAM]: LOOP_OFF };
      const { wrapper } = setup();
      expect(wrapper.find(BasePlayer)).toHaveProp({ loop: false });
    });

    it('use default with unknown LOOP_PARAM value', () => {
      mockUseFragment.mockImplementationOnce(() => mockClip);
      mockCurrentQuery = { [LOOP_PARAM]: random.alphaNumeric() };
      const { wrapper } = setup();
      expect(wrapper.find(BasePlayer)).toHaveProp({ loop: true });
    });
  });
});
