import { PlayerType, useStreamAnalytics } from 'pulsar';
import type { FC } from 'react';
import { graphql, useFragment } from 'react-relay/hooks';
import { useStaticEnvironment } from 'tachyon-environment';
import { channelPulsarParams } from 'tachyon-page-utils';
import type { LoadedEventData } from '../../controller';
import { BasePlayer } from '../BasePlayer';
import { useOnloadMetadataEmitter } from '../useOnLoadMetadataEmitter';
import { useTrackingEvent } from '../useTrackingEvent';
import type {
  StreamPlayerManager_channel,
  StreamPlayerManager_channel$key,
} from './__generated__/StreamPlayerManager_channel.graphql';

const streamPlayerManagerFragment = graphql`
  fragment StreamPlayerManager_channel on User {
    id
    login
    roles {
      isPartner
    }
    self {
      subscriptionBenefit {
        id
      }
    }
    stream {
      id
      createdAt
      game {
        id
        name
      }
    }
  }
`;

export type StreamPlayerManagerProps = {
  channel: StreamPlayerManager_channel$key;
};

// istanbul ignore next: trivial
export function streamMetadata(
  channel: StreamPlayerManager_channel,
): LoadedEventData {
  return {
    broadcastTimestamp: channel.stream?.createdAt ?? null,
    type: 'stream',
  };
}

// istanbul ignore next: trivial
export const StreamPlayerManager: FC<StreamPlayerManagerProps> = ({
  channel: channelRef,
}) => {
  const channel = useFragment(streamPlayerManagerFragment, channelRef);
  useOnloadMetadataEmitter(streamMetadata(channel));

  const {
    client: { sessionID },
  } = useStaticEnvironment();

  const onTrackingEvent = useTrackingEvent();
  useStreamAnalytics({
    channel,
    currentUser: null,
    onTrackingEvent,
    tracking: {
      playerType: PlayerType.Pulsar,
      sessionId: sessionID,
    },
  });

  return <BasePlayer params={channelPulsarParams(channel)} />;
};

StreamPlayerManager.displayName = 'StreamPlayerManager';
