import { PlayerType, useVodAnalytics } from 'pulsar';
import type { FC } from 'react';
import { graphql, useFragment } from 'react-relay/hooks';
import { useStaticEnvironment } from 'tachyon-environment';
import { useStartTime, vodPulsarParams } from 'tachyon-page-utils';
import { START_FROM_TIME_PARAM } from '../../../config';
import type { LoadedEventData } from '../../controller';
import { BasePlayer } from '../BasePlayer';
import { useOnloadMetadataEmitter } from '../useOnLoadMetadataEmitter';
import { useTrackingEvent } from '../useTrackingEvent';
import type {
  VodPlayerManager_video,
  VodPlayerManager_video$key,
} from './__generated__/VodPlayerManager_video.graphql';

const vodPlayerManagerFragment = graphql`
  fragment VodPlayerManager_video on Video {
    broadcastType
    createdAt
    id
    game {
      name
    }
    lengthSeconds
    owner {
      id
      login
      roles {
        isPartner
      }
      self {
        subscriptionBenefit {
          id
        }
      }
    }
  }
`;

export type VodPlayerManagerProps = {
  video: VodPlayerManager_video$key;
};

// istanbul ignore next: trivial
export function vodMetadata(video: VodPlayerManager_video): LoadedEventData {
  return {
    broadcastTimestamp: video.createdAt,
    duration: video.lengthSeconds,
    type: 'vod',
  };
}

// istanbul ignore next: trivial
export const VodPlayerManager: FC<VodPlayerManagerProps> = ({
  video: videoRef,
}) => {
  const video = useFragment(vodPlayerManagerFragment, videoRef);
  const startTime = useStartTime(START_FROM_TIME_PARAM);

  useOnloadMetadataEmitter(vodMetadata(video));

  const {
    client: { sessionID },
  } = useStaticEnvironment();

  const onTrackingEvent = useTrackingEvent();
  useVodAnalytics({
    currentUser: null,
    onTrackingEvent,
    tracking: {
      playerType: PlayerType.Pulsar,
      sessionId: sessionID,
    },
    video,
  });

  return (
    <BasePlayer
      params={vodPulsarParams({
        vod: video,
      })}
      startTime={startTime}
    />
  );
};

VodPlayerManager.displayName = 'VodPlayerManager';
