import type { ParsedUrlQuery } from 'querystring';
import type { LinkProps as NextLinkProps } from 'next/link';
import NextLink from 'next/link';
import type {
  AnchorHTMLAttributes,
  DetailedHTMLProps,
  PropsWithChildren,
} from 'react';
import { convertGqlIdValuesToUnsafe } from 'tachyon-relay';
import type { AsPathForRouteNameParams } from '../../linkPartsForRouteName';
import {
  isDynamicParams,
  linkPartsForRouteName,
} from '../../linkPartsForRouteName';
import type { RouteName } from '../../routes';

type NextLinkPublicProps = Omit<NextLinkProps, 'as' | 'href'>;

type TachyonLinkProps<T extends RouteName> = PropsWithChildren<
  AsPathForRouteNameParams<T> &
    NextLinkPublicProps & {
      anchorProps?: DetailedHTMLProps<
        AnchorHTMLAttributes<HTMLAnchorElement>,
        HTMLAnchorElement
      >;
      query?: ParsedUrlQuery;
    }
>;

/**
 * A wrapper around Next's Link component that provides type safe link generations.
 * All props from the original are available except "href" and "as" which are constructed
 * using "route" (in all cases) and "routeParams" (in the case of a dynamic route).
 *
 * Static Route:
 * <TachyonLink route={RouteName.Homepage}>Home</TachyonLink>
 *
 * Dynamic Route:
 * <TachyonLink route={RouteName.GameDirectory} routeParams={{ game: 'fortnite' }}>
 *   Fortnite
 * </TachyonLink>
 *
 * Query Params:
 * <TachyonLink route={RouteName.Homepage} query={{ tt_content: 'foo' }}>
 *   Home
 * </TachyonLink>
 */
export function TachyonLink<T extends RouteName>(
  props: TachyonLinkProps<T>,
): JSX.Element {
  // The use of "any" allows us to bypass the fact that TypeScript can't be sure
  // at this point as to what "params" requires as arguments due to type complexity.
  // However, Link's public prop API is type safe so we know for certain that if
  // routeParams are present, they are correct.
  const params: any = isDynamicParams(props)
    ? {
        route: props.route,
        // TODO: it is possible to have null params here, not sure why?
        // eslint-disable-next-line @typescript-eslint/no-unsafe-assignment
        routeParams:
          props.routeParams &&
          convertGqlIdValuesToUnsafe(props.routeParams as any),
      }
    : { route: props.route };

  // eslint-disable-next-line @typescript-eslint/no-unsafe-argument
  const { as, href } = linkPartsForRouteName(params, props.query);

  const linkProps: NextLinkPublicProps = {
    passHref: true,
    prefetch: props.prefetch,
    replace: props.replace,
    scroll: props.scroll,
    shallow: props.shallow,
  };

  return (
    <NextLink {...linkProps} as={as} href={href}>
      <a {...props.anchorProps}>{props.children}</a>
    </NextLink>
  );
}

TachyonLink.displayName = 'TachyonLink';
