/**
 * A list of all route names in the app. This list must manually be updated when a new
 * page is added to the application. The values map to the route's equivalent Spade "location".
 */
export enum RouteName {
  Channel = 'channel',
  Clip = 'clip',
  Error = 'error',
  HealthCheck = 'health_check',
  Homepage = 'homepage',
  NotFound = 'not_found',
  Vod = 'vod',
}

/**
 * A mapping of Route names to their Next equivalent static "pathname". These
 * map directly to the folder structure of src/pages/*.
 */
export const RouteLinkPathnames: { [key in RouteName]: string } = {
  [RouteName.Clip]: '/clip/[slug]',
  [RouteName.Vod]: '/videos/[videoId]',
  [RouteName.Error]: '/_error',
  [RouteName.HealthCheck]: '/_debug/running',
  [RouteName.Homepage]: '/',
  [RouteName.NotFound]: '/not-found',
  // Channel must be last in this list to allow for proper matching behavior
  [RouteName.Channel]: '/[login]',
};

export type RouteLinkParams = {
  as?: string;
  href: string;
};

/**
 * A mapping of route names to a matching link builder function. The arguments
 * for these functions also serve as the definition for which "routeParams" to
 * provide to the app's "Link" component.
 *
 * DO NOT USE DIRECTLY: Use `Link` ("src/components/framework/Link").
 *
 * Rules:
 * 1) All functions must return an object of { href: string; as?: string }.
 * 1) Static routes should take no arguments in their function and return only an "href" field.
 * 2) Dynamic routes should return both an "href" (path with placeholder) and "as" (path with dynamic value) field.
 */
export const RouteLinkBuilders = {
  [RouteName.Clip]: ({ slug }: { slug: string }): RouteLinkParams => ({
    as: `/clip/${slug}`,
    href: RouteLinkPathnames[RouteName.Clip],
  }),
  [RouteName.Error]: (): RouteLinkParams => ({
    href: RouteLinkPathnames[RouteName.Error],
  }),
  [RouteName.Vod]: ({ videoId }: { videoId: string }): RouteLinkParams => ({
    as: `/videos/${videoId}`,
    href: RouteLinkPathnames[RouteName.Vod],
  }),
  [RouteName.HealthCheck]: (): RouteLinkParams => ({
    href: RouteLinkPathnames[RouteName.HealthCheck],
  }),
  [RouteName.Homepage]: (): RouteLinkParams => ({
    href: RouteLinkPathnames[RouteName.Homepage],
  }),
  [RouteName.NotFound]: (): RouteLinkParams => ({
    href: RouteLinkPathnames[RouteName.NotFound],
  }),
  [RouteName.Channel]: ({ login }: { login: string }): RouteLinkParams => ({
    as: `/${login}`,
    href: RouteLinkPathnames[RouteName.Channel],
  }),
};

/**
 * Converts a custom RouteName to a Next pathname.
 * Returns the NotFound pathname if there is no matching route.
 */
export function pathnameFromRouteName(name: RouteName): string {
  return RouteLinkPathnames[name] || RouteLinkPathnames[RouteName.NotFound];
}

/**
 * Converts a Next pathname to a RouteName. Returns NotFound if there is no matching path.
 */
export function routeNameFromPathname(pathname: string): RouteName {
  const routeName = Object.keys(RouteLinkPathnames).find(
    (route) => RouteLinkPathnames[route as RouteName] === pathname,
  ) as RouteName | undefined;

  return routeName ?? RouteName.NotFound;
}
