import { datatype, internet, random } from 'faker';
import type {
  TachyonRequest,
  TachyonRequestExtension,
  TachyonResponse,
} from 'tachyon-next-types';
import { createDeviceIDOnServer } from 'tachyon-server-utils';
import { GLOBAL_COOKIE_NAMES } from 'tachyon-utils';
import { createAppRequestHandler } from '.';

const mockSpadeUrl = 'https://spade.twitch.tv/';

jest.mock('tachyon-server-utils', () => ({
  ...jest.requireActual('tachyon-server-utils'),
  createDeviceIDOnServer: jest.fn(() => 'tach-fake-device-id'),
}));
const mockCreateDeviceIDOnServer = createDeviceIDOnServer as jest.Mock;

describe(createAppRequestHandler, () => {
  const mockContext = {
    appEnvironment: 'development',
    dynamicSettings: {
      spadeUrl: mockSpadeUrl,
    },
  } as TachyonRequestExtension;
  const deviceId = datatype.uuid();
  const path = `/${random
    .words(datatype.number({ max: 4 }))
    .replace(/ /g, '/')}`;

  let req: TachyonRequest;
  let res: TachyonResponse;
  const next = () => undefined;
  const mockNextRequestHandler = jest.fn();
  const handler = createAppRequestHandler(mockContext, mockNextRequestHandler);

  beforeEach(() => {
    mockNextRequestHandler.mockReset();

    req = {
      cookies: {
        [GLOBAL_COOKIE_NAMES.DEVICE_ID_COOKIE_NAME]: deviceId,
      },
      path,
      url: internet.url(),
    } as any;

    res = { cookie: jest.fn() } as any;
  });

  it('creates a request handler function that adds static properties to req and calls next request handler', () => {
    expect(handler).toBeInstanceOf(Function);

    handler(req, res, next);
    expect(req.tachyon).toEqual(
      expect.objectContaining({
        appEnvironment: 'development',
        dynamicSettings: { spadeUrl: mockSpadeUrl },
      }),
    );

    expect(mockNextRequestHandler).toHaveBeenCalledWith(
      req,
      res,
      expect.objectContaining({ href: expect.stringContaining(req.url) }),
    );
  });

  describe('device id', () => {
    it('does not make device id cookies', () => {
      delete req.cookies[GLOBAL_COOKIE_NAMES.DEVICE_ID_COOKIE_NAME];
      handler(req, res, next);

      expect(mockCreateDeviceIDOnServer).not.toHaveBeenCalled();
    });
  });
});
