import { resolve } from 'path';
import { URL } from 'url';
import cookieParser from 'cookie-parser';
import express from 'express';
import next from 'next';
import open from 'open';
import { fetchDynamicSettings } from 'tachyon-dynamic-settings';
import {
  getTachyonEnvVar,
  tachyonServerMiddleware,
} from 'tachyon-server-utils';
import { ensureStringArray } from 'tachyon-utils';
import type { ValenceDynamicSettings } from '../config';
import {
  FLUME_ALLOWED_EVENTS_FALLBACK,
  FLUME_ALLOWED_PROPERTIES_FALLBACK,
} from '../config';
import { RouteName, dynamicPathnames, pathnameFromRouteName } from '../routing';
import { createAppRequestHandler } from './appHandler';
import { isDevelopment } from './utils';

// polyfill WHATWG URL
// eslint-disable-next-line @typescript-eslint/no-unsafe-member-access
(global as any).URL = URL;

const PORT = process.env.PORT ? parseInt(process.env.PORT, 10) : 3300;
const dev = isDevelopment();

// from `dist/server/index.js` to dir containing `.next/` & `messages/`
const appDir = dev ? undefined : resolve(__dirname, '../..');

// 15 minutes
const DYNAMIC_SETTINGS_TTL = 15 * 60 * 1000;

const app = next({ dev, dir: appDir });
const nextRequestHandler = app.getRequestHandler();
const appEnvironment = getTachyonEnvVar();

// istanbul ignore next: trivial + type-based enforcement
const fetchValenceDynamicSettings =
  fetchDynamicSettings<ValenceDynamicSettings>({
    app: 'valence',
    appEnvironment,
    appGroup: 'tachyon',
    processor: ({
      flume_allowed_events,
      flume_allowed_properties,
      ...settings
    }) => {
      const flumeAllowedEvents = ensureStringArray(
        flume_allowed_events,
        FLUME_ALLOWED_EVENTS_FALLBACK,
      );
      const flumeAllowedProperties = ensureStringArray(
        flume_allowed_properties,
        FLUME_ALLOWED_PROPERTIES_FALLBACK,
      );

      return {
        ...settings,
        flumeAllowedEvents,
        flumeAllowedProperties,
      };
    },
  });

// istanbul ignore next: trivial
fetchValenceDynamicSettings()
  .then((dynamicSettings) => {
    const ctx = {
      appEnvironment,
      dynamicSettings,
    };

    setInterval(() => {
      fetchValenceDynamicSettings().then((settings) => {
        ctx.dynamicSettings = settings;
      });
    }, DYNAMIC_SETTINGS_TTL).unref();

    return ctx;
  })
  .then((ctx) =>
    app.prepare().then(() => {
      const server = express();
      // Necessary until https://github.com/vercel/next.js/issues/17096 is fixed
      server.use(
        tachyonServerMiddleware({
          badRequestMiddleware: {
            badRequestRedirectPath: pathnameFromRouteName(RouteName.NotFound),
            dynamicPathnames: dynamicPathnames(),
          },
        }),
      );
      server.use(cookieParser());
      // tell express to populate `req` convenience attributes from X-Forwarded-* headers
      // http://expressjs.com/en/4x/api.html#req.hostname
      server.enable('trust proxy');

      const appRequestHandler = createAppRequestHandler(
        ctx,
        nextRequestHandler,
      );
      // api routes are the only routes that need to handle POST requests, so we
      // make this explicit to minimize our exposed surface area
      server.post('/api/*', appRequestHandler);
      server.get('*', appRequestHandler);

      server.listen(PORT, () => {
        if (dev) {
          const devUrl = 'https://localhost.valence.tv.twitch.tv';
          console.log(`\nListening at ${devUrl}\n`);
          open(devUrl);
        }
      });
    }),
  )
  .catch((error: Error) => {
    console.error(error.stack);
    process.exit(1);
  });
