# Progressive Web Application

This document outlines PWA specifics in Mobile Web.

### Display Mode

We set the `display-mode` for the PWA in our
[manifest](https://developer.mozilla.org/en-US/docs/Web/Manifest) to
`standalone`, which most closely resembles a native experience. Here is an
[overview of display mode visual representations](https://superpwa.com/doc/web-app-manifest-display-modes/)

### Platform

When Mobile Web is experienced as a PWA the platform is still set to
`mobile_web`.

There was discussion to set a new platform value of `pwa_web` in
[PWA Platform Spec](https://docs.google.com/document/d/1nsaPLY2TabYku4IYEdHv7gLWeqfcE0lzxLXM4e-1_V0/edit#heading=h.uae6wug3njjy)
but data did not feel that PWAs should be distinguished as a separate platform.

We detect when mobile web is being viewed as a PWA experience by inspecting the
[display-mode](https://developer.mozilla.org/en-US/docs/Web/CSS/@media/display-mode).
See `src/utils/isStandaloneApp/index.ts` in TMW.

### App Shell

A common PWA architecture pattern (and the one employed by mobile web) is an
[app shell](https://developer.mozilla.org/en-US/docs/Web/Progressive_web_apps/App_structure#App_shell),
where the app shell is a bundle of static assets that render a minimal UI and
manages dynamic content. This app shell can be served from the browser cache
while dynamic content is loading or while offline.

The functionality necessary for an app shell is provided by Next.js. We serve a
minimal page `/app-shell` which is cached during service worker installation.
Once installed, all future page requests will be handled by the app shell until
the service worker receives an update.

The page served by `/app-shell` renders a client side redirect to the Homepage,
so the existence of the app shell is transparent to the user, and all pages can
continue to rely on the existing data fetching architecture built on
`getInitialProps`. The page itself does not render any content.

### Offline

Mobile web tracks actions taken by the user even when the user’s device is
offline. When the network is unavailable, spade events are queued on the device
and sent once the network is available again. This is implemented via
[Background Sync](https://developers.google.com/web/updates/2015/12/background-sync).

When the GraphQL request for a page fails, we will render an offline page in
place of the requested page. If the page implements `OfflinePage` as a property,
that component will be used as the offline page. Otherwise, we'll fallback to
rendering `DefaultOfflinePage`.

When an offline page is viewed, the corresponding `pageview` event will have
`client_offline` set to true. This can be used to see which pages users view
most in an offline setting. The `pageview` is otherwise identical to the online
equivalent for the page (though some fields that rely on GraphQL may be
missing).
[Offline Tracking Spec](https://docs.google.com/document/d/1bonieQzphBR2i3KGjTZ6gOVg3R17kv-jZ8Qi4uOwmyA/edit#)
for more context.

NOTE: To test the offline experience, you'll need to run `yarn prodlike`. The
[service worker](../../../apps/tomorrow/processes/service-worker-development.md')
is required to power the offline experience, and it is only run in `prodlike`.
During installation the service worker installs all of the JS chunks generated
by Next.js. Without this, the client will make network requests for JS chunks
that it does not have (they are lazy loaded), which would fail to fetch in an
offline setting and cause the browser to show a default network error page.

### Helpful Resources

[PWA Advantages](https://developer.mozilla.org/en-US/docs/Web/Progressive_web_apps/Introduction#Advantages_of_web_applications)(~5min
read)

[MDN PWA Introduction](https://developer.mozilla.org/en-US/docs/Web/Progressive_web_apps/Introduction)(~45min
read)

[Product Spec](https://docs.google.com/document/d/1R2jUmVvNS0hE1qz5mOk9YfSeaLQQF_CsrygXfrGDiNQ/edit#heading=h.87grz1latyoh)

[Technical Spec](https://docs.google.com/document/d/1N66T2HOLRbvKLSALeNflB868NActeo_gCCftuOCM92E/edit)
