package tv.twitch.starshot64.analytics

import org.json.JSONObject

/**
 * Base class for tracking events.
 */
abstract class TrackingEvent(val context: String, private val subContext: String? = null) {

  var clientApp: String? = null
  var language: String? = null
  var osName: String? = null
  var osVersion: String? = null
  var twitchAppVersion: String? = null
  var twitchDeviceId: String? = null
  var twitchPlatform: String? = null
  private val payload: JSONObject = JSONObject()

  fun setPayloadValue(key: String, value: JSONObject) {
    payload.put(key, value)
  }
  fun setPayloadValue(key: String, value: Int) {
    payload.put(key, value)
  }
  fun setPayloadValue(key: String, value: Float) {
    payload.put(key, value)
  }
  fun setPayloadValue(key: String, value: String) {
    payload.put(key, value)
  }

  /**
   * Builds a JSON object from the event data.
   */
  fun toJson(): JSONObject {
    val json = JSONObject()
    json.put("event", "native_shell_event")
    json.put("context", context)

    if (subContext != null) {
      json.put("subcontext", subContext)
    }

    val properties = propertiesToJson()
    json.put("properties", properties)

    return json
  }

  /**
   * Creates a JSON object from the properties.  This should be overridden if the concrete event
   * has additional properties.
   */
  protected fun propertiesToJson(): JSONObject {
    val json = JSONObject()

    json.put("app_version", twitchAppVersion)
    json.put("client_app", clientApp)
    json.put("device_id", twitchDeviceId)
    json.put("language", language)
    json.put("os_name", osName)
    json.put("os_version", osVersion)
    json.put("platform", twitchPlatform)

    if (payload.length() > 0) {
      json.put("payload", payload)
    }

    return json
  }
}

class NativeShellLaunchTrackingEvent : TrackingEvent("launch")
class NativeShellLoginTrackingEvent : TrackingEvent("login")
class NativeShellLogoutTrackingEvent : TrackingEvent("logout")
class NativeShellErrorTrackingEvent(private val error: String) :
  TrackingEvent("error", error)
