package tv.twitch.starshot64.analytics

import android.content.Context
import android.os.Build
import kotlin.collections.ArrayList
import okhttp3.Interceptor
import okhttp3.OkHttpClient
import okhttp3.Response
import org.json.JSONObject
import timber.log.Timber
import tv.twitch.starshot64.BuildConfig
import tv.twitch.starshot64.util.BackoffTimer
import tv.twitch.starshot64.util.getDeviceId
import tv.twitch.starshot64.util.getLocaleString

/**
 * Batch events for this amount of time before sending them during normal, non-error conditions.
 */
private const val DEFAULT_BATCH_TIME_MILLISECONDS: Long = 1 * 1000 // 1 second

/**
 * The maximum number of milliseconds to hold events for in the worst case of not being able
 * to successfully contact Twitch.
 */
private const val MAX_BATCH_BACKOFF_TIME_MILLISECONDS: Long = 5 * 60 * 1000 // 5 minutes

/**
 * The maximum number of milliseconds to jitter when attempting to resend failed events.
 */
private const val MAX_BATCH_BACKOFF_JITTER_MILLISECONDS: Long = 10 * 1000 // 10 seconds

/**
 * The absolute max number of events that can be in the queue before we discard older ones.
 */
private const val MAX_QUEUE_SIZE = 40

/**
 * The data that must be protected from concurrent access.
 */
class ProtectedData {
  /**
   * The list of events that have yet to be sent.
   */
  val queue: ArrayList<JSONObject> = ArrayList()

  var httpClient: OkHttpClient? = null
  var batchTimer = BackoffTimer(
    "SpadeBatch",
    MAX_BATCH_BACKOFF_TIME_MILLISECONDS,
    MAX_BATCH_BACKOFF_JITTER_MILLISECONDS,
    ::flush
  )

  var spadeUrlRotator: SpadeUrlRotator? = null
  var inFlightBatch: SpadeEventBatch? = null
  var initialSpadeUrlFetchComplete = false
}

/**
 * The data that must be updated atomically.  This instance will be used as the lock to protect
 * itself.
 */
private val gProtectedData = ProtectedData()

private val clientApp = "starshot"
private var language = ""
private val osName = "Android"
private val osVersion = Build.VERSION.RELEASE
private var twitchAppVersion = BuildConfig.VERSION_NAME
private var twitchDeviceId = ""
private val twitchPlatform = BuildConfig.TWITCH_PLATFORM

/**
 * Sets up the tracking subsystem.
 */
fun initializeTracking(context: Context) {
  Timber.d("initializeTracking")

  twitchDeviceId = getDeviceId(context)
  language = getLocaleString()

  synchronized(gProtectedData) {
    // Already initialized
    if (gProtectedData.httpClient != null) {
      return
    }

    // Create the HTTP service
    val http = OkHttpClient.Builder()
      .addInterceptor(SpadeHeadersInterceptor(context))
      .build()
    gProtectedData.httpClient = http

    // Setup the URL rotator
    gProtectedData.spadeUrlRotator = SpadeUrlRotator(http, ::handleSpadeUrlFetchComplete)
  }
}

/**
 * Sets the common properties from system values on the given event.
 */
fun setCommonProperties(event: TrackingEvent) {
  event.clientApp = clientApp
  event.language = language
  event.osName = osName
  event.osVersion = osVersion.toString()
  event.twitchAppVersion = twitchAppVersion
  event.twitchDeviceId = twitchDeviceId
  event.twitchPlatform = twitchPlatform
}

private fun handleSpadeUrlFetchComplete(succeeded: Boolean) {
  // NOTE: We ignore a failure to obtain the URL and use the default
  synchronized(gProtectedData) {
    // Open the gates to allow Spade events to be sent now that we have the URL
    if (!gProtectedData.initialSpadeUrlFetchComplete) {
      gProtectedData.batchTimer.start(DEFAULT_BATCH_TIME_MILLISECONDS)
      gProtectedData.initialSpadeUrlFetchComplete = true
    }
  }
}

/**
 * Enqueues the given tracking event.
 */
fun sendTrackingEvent(event: TrackingEvent) {
  Timber.d("sendTrackingEvent")
  setCommonProperties(event)
  synchronized(gProtectedData) {
    gProtectedData.queue.add(event.toJson())
    pruneQueue()

    // Still waiting for the initial Spade URL fetch
    if (!gProtectedData.initialSpadeUrlFetchComplete) {
      return
    }

    // A request is already in progress so do nothing
    if (gProtectedData.inFlightBatch != null) {
      return
    }

    // Schedule flush if not already scheduled
    if (!gProtectedData.batchTimer.isStarted()) {
      gProtectedData.batchTimer.start(DEFAULT_BATCH_TIME_MILLISECONDS)
    }
  }
}

/**
 * If we have too many events queued up throw out the oldest.
 */
private fun pruneQueue() {
  while (gProtectedData.queue.size > MAX_QUEUE_SIZE) {
    gProtectedData.queue.removeAt(0)
  }
}

/**
 * Prepares and sends an event batch.
 * This assumes the data is already locked.
 */
private fun flush() {
  // Nothing to do
  if (gProtectedData.queue.isEmpty()) {
    return
  }

  // Build the list of events to send
  val inFlight = ArrayList<JSONObject>()
  for (i in 0 until gProtectedData.queue.size) {
    val obj = gProtectedData.queue[i]
    inFlight.add(obj)
  }
  gProtectedData.queue.clear()

  // Kick off the send
  gProtectedData.inFlightBatch = SpadeEventBatch(
    gProtectedData.httpClient!!,
    gProtectedData.spadeUrlRotator!!.activeUrl(),
    inFlight,
    ::handleBatchSendFinished
  )
}

private fun handleBatchSendFinished(succeeded: Boolean) {
  if (succeeded) {
    // Schedule the next batch
    if (gProtectedData.queue.isNotEmpty()) {
      gProtectedData.batchTimer.start(DEFAULT_BATCH_TIME_MILLISECONDS)
    }
  } else {
    // Add the events back into the front of the queue to be sent again next time
    gProtectedData.queue.addAll(0, gProtectedData.inFlightBatch!!.events)

    pruneQueue()

    // Set a backoff timer so we can try again soon
    gProtectedData.batchTimer.startBackoff()
  }

  gProtectedData.inFlightBatch = null
}

private class SpadeHeadersInterceptor(val context: Context) : Interceptor {
  val deviceId = getDeviceId(context)

  override fun intercept(chain: Interceptor.Chain): Response {
    val requestBuilder = chain.request().newBuilder()
      .addHeader("Client-ID", BuildConfig.TWITCH_CLIENT_ID)
      .addHeader("X-Device-Id", deviceId)

    return chain.proceed(requestBuilder.build())
  }
}
