package tv.twitch.starshot64.analytics

import android.util.Base64
import java.io.IOException
import java.nio.charset.StandardCharsets
import okhttp3.Call
import okhttp3.Callback
import okhttp3.MediaType
import okhttp3.OkHttpClient
import okhttp3.Request
import okhttp3.RequestBody
import okhttp3.Response
import org.json.JSONArray
import org.json.JSONObject

class SpadeEventBatch(
  private val httpClient: OkHttpClient,
  private val spadeUrl: String,
  val events: ArrayList<JSONObject>,
  sendFinishedCallback: (succeeded: Boolean) -> Unit
) {

  private val responseHandler = ResponseHandler(sendFinishedCallback)

  init {
    // Kick off the request immediately
    val base64 = generateBase64(events)
    makeRequest(base64)
  }

  /**
   * Generate the base64 data from the JSON data.
   */
  private fun generateBase64(events: ArrayList<JSONObject>): String {
    // Build the JSON payload of all events we can send at once
    val arr = JSONArray()

    for (i in 0 until events.size) {
      arr.put(i, events[i])
    }

    // Serialize to a Base64 string
    val serialized = arr.toString().toByteArray()
    val bytes = Base64.encode(serialized, Base64.NO_WRAP)

    return String(bytes, StandardCharsets.UTF_8)
  }

  /**
   * Make the HTTP request with the given encoded data.
   */
  private fun makeRequest(base64EncodedData: String) {
    try {
      val mediaType = MediaType.get("application/x-www-form-urlencoded")
      val body = RequestBody.create(mediaType, "data=$base64EncodedData")

      val request = Request.Builder()
        .url(spadeUrl)
        .method("POST", body)
        .build()

      // Make the request
      httpClient.newCall(request).enqueue(responseHandler)
    } catch (ex: Exception) {
      responseHandler.failed()
    }
  }

  private class ResponseHandler(
    private val sendFinishedCallback: (succeeded: Boolean) -> Unit
  ) : Callback {

    override fun onResponse(call: Call, response: Response) {
      val code = response.code()
      if (code in 200..299) {
        succeeded()
      } else {
        failed()
      }
    }

    override fun onFailure(call: Call, e: IOException) {
      failed()
    }

    private fun succeeded() {
      sendFinishedCallback(true)
    }

    fun failed() {
      sendFinishedCallback(false)
    }
  }
}
