package tv.twitch.starshot64.analytics

import java.io.IOException
import okhttp3.Call
import okhttp3.Callback
import okhttp3.OkHttpClient
import okhttp3.Request
import okhttp3.Response
import timber.log.Timber
import tv.twitch.starshot64.util.BackoffTimer

/**
 * In order to get around ad blockers preventing spade traffic from reaching our backend, we have
 * a rotating endpoint that should be used for sending events.  This is the endpoint that gives the
 * daily rotating URL.
 */
private const val SPADE_ROTATOR_URI = "https://science-output.s3.amazonaws.com/spade-uri"

/**
 * The default URL to use if we are unable to obtain a rotated Spade URL.
 */
private const val DEFAULT_SPADE_URL = " https://spade.twitch.tv"

/**
 * The frequency in which to update the Spade URL.
 */
private const val URL_ROTATION_FREQUENCY_TIME: Long = 24 * 60 * 60 * 1000 // 24 hours

/**
 * The max backoff when scheduling retries.
 */
private const val MAX_BACKOFF_TIME_MILLISECONDS: Long = 4 * 60 * 60 * 1000 // 4 hours

/**
 * The maximum number of milliseconds to jitter add when attempting to fetch the url again.
 */
private const val MAX_BACKOFF_JITTER_MILLISECONDS: Long = 10 * 60 * 1000 // 10 minutes

class SpadeUrlRotator(
  private val httpClient: OkHttpClient,
  private val updateFinishedCallback: (succeeded: Boolean) -> Unit
) {

  private val lock = Object()
  private val responseHandler = ResponseHandler(::handleSuccessResponse, ::handleFailedResponse)
  private var timer = BackoffTimer(
    "SpadeUrlRotator",
    MAX_BACKOFF_TIME_MILLISECONDS,
    MAX_BACKOFF_JITTER_MILLISECONDS,
    ::update
  )
  private var spadeUrl = DEFAULT_SPADE_URL

  init {
    // Kick off an update right away
    update()
  }

  /**
   * Retrieves the active Spade url.
   */
  fun activeUrl(): String {
    return spadeUrl
  }

  /**
   * Perform the URL fetch now.
   */
  private fun update() {
    timer.cancel()
    makeRequest()
  }

  private fun makeRequest() {
    try {
      val request = Request.Builder()
        .url(SPADE_ROTATOR_URI)
        .method("GET", null)
        .build()

      // Make the request
      httpClient.newCall(request).enqueue(responseHandler)
    } catch (ex: Exception) {
      timer.startBackoff()
    }
  }

  private fun handleSuccessResponse(url: String) {
    synchronized(lock) {
      if (url.startsWith("//")) {
        spadeUrl = "https:$url"
      } else {
        spadeUrl = url
      }
      timer.start(URL_ROTATION_FREQUENCY_TIME)
    }

    // Notify the external listener
    updateFinishedCallback(true)
  }

  private fun handleFailedResponse(detail: String, statusCode: Int) {
    synchronized(lock) {
      // Send a spade event the first time the error occurs
      if (!timer.isBackingOff()) {
        val event = NativeShellErrorTrackingEvent("spade-url-rotation")
        event.setPayloadValue("detail", detail)
        event.setPayloadValue("status-code", statusCode)
        sendTrackingEvent(event)
      }

      timer.startBackoff()
    }

    // Notify the external listener
    updateFinishedCallback(false)
  }

  private class ResponseHandler(
    private val succeeded: (url: String) -> Unit,
    private val failed: (detail: String, statusCode: Int) -> Unit
  ) : Callback {

    override fun onResponse(call: Call, response: Response) {
      val code = response.code()
      if (code in 200..299) {
        val body = response.body()
        if (body != null) {
          succeeded(body.string())
        } else {
          failed("empty-body", code)
          Timber.e("Got empty body from spade url rotator")
        }
      } else {
        failed("request-failed", code)
        Timber.e("Got unexpected status code from spade url rotator: $code")
      }
    }

    override fun onFailure(call: Call, e: IOException) {
      failed("request-did-not-complete", 0)
      Timber.e(e)
    }
  }
}
