package tv.twitch.starshot64.app

import android.annotation.SuppressLint
import android.content.Context
import android.content.Intent
import android.content.pm.ApplicationInfo
import android.os.Bundle
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import android.webkit.CookieManager
import android.webkit.WebSettings
import android.webkit.WebView
import androidx.appcompat.app.AppCompatActivity
import java.net.URLDecoder
import timber.log.Timber
import tv.twitch.starshot64.BuildConfig
import tv.twitch.starshot64.R
import tv.twitch.starshot64.media.IMediaSessionListener
import tv.twitch.starshot64.media.TwitchMediaSession
import tv.twitch.starshot64.recommendations.getRecommendationsStrategy
import tv.twitch.starshot64.util.crossFadeViews
import tv.twitch.starshot64.util.getCredentials
import tv.twitch.starshot64.util.getDeviceId
import tv.twitch.starshot64.util.getSecondsFromMs

/**
 * Duration of the cross fade between splash and web view.
 */
private const val crossFadeDurationMillis: Long = 500

class StarshotActivity : AppCompatActivity() {

  private lateinit var rootContainer: ViewGroup
  private lateinit var loadingView: View
  private var webView: WebView? = null
  private var nextWebViewId: Long = 0

  private val mediaSession: TwitchMediaSession = TwitchMediaSession()

  private val mainState: MainState = MainState(this)
  private val errorState: ErrorState = ErrorState()

  override fun onCreate(savedInstanceState: Bundle?) {
    super.onCreate(savedInstanceState)

    Timber.d("StarshotActivity.onCreate")

    // Load layout xml
    setContentView(R.layout.starshot_activity)

    // Pull individual views from the layout
    rootContainer = findViewById(R.id.rootContainer)
    loadingView = findViewById(R.id.splashview)

    mediaSession.onActivityCreate(getApplicationContext())

    var launchMain = true

    // Determine where to go
    if (0 != applicationInfo.flags and ApplicationInfo.FLAG_DEBUGGABLE) {
      // Go directly to the error page in LaserArray
      val debugLaserArray = intent.getStringExtra("debug-laser-array")
      if (debugLaserArray == "1") {
        launchMain = false
        loadErrorScreen()
      }
    }

    // Launch the webview
    if (launchMain) {
      loadStarshot(intent, forceReload = true)
    }
  }

  override fun onStart() {
    super.onStart()
    Timber.d("StarshotActivity.onStart")

    mediaSession.onActivityStart()

    getRecommendationsStrategy().handleActivityStarted(this)
  }

  override fun onPause() {
    super.onPause()
    Timber.d("StarshotActivity.onPause")

    mediaSession.onActivityPause()
  }

  override fun onResume() {
    super.onResume()
    Timber.d("StarshotActivity.onResume")

    mediaSession.onActivityResume()
  }

  override fun onStop() {
    super.onStop()
    Timber.d("StarshotActivity.onStop")

    mediaSession.onActivityStop()
  }

  override fun onDestroy() {
    super.onDestroy()
    Timber.d("StarshotActivity.onDestroy")

    mediaSession.onActivityDestroy()
  }

  override fun onNewIntent(intent: Intent) {
    super.onNewIntent(intent)
    Timber.d("StarshotActivity.onNewIntent")

    loadStarshot(intent, forceReload = false)
  }

  override fun onBackPressed() {
    Timber.d("StarshotActivity.onBackPressed")
    val view = webView
    if (view != null) {
      if (view.canGoBack()) {
        view.goBack()
      } else {
        super.onBackPressed()
      }
    }
  }

  private fun showLoadingScreen() {
    Timber.d("StarshotActivity.showLoadingScreen")
    loadingView.visibility = View.VISIBLE
    loadingView.alpha = 1f
    loadingView.requestFocus()
  }

  private fun createWebViewClient(webView: WebView, nativeProxy: NativeProxyBase):
    WebViewClientBase {

      val firstLoadCallback = { succeeded: Boolean ->
        rootContainer.post {
          // Hide the loading screen and show the webview
          if (succeeded) {
            webViewLoaded(webView)
          }
          // Show the offline error page
          else {
            loadErrorScreen()
          }
        }
        Unit
      }

      val client = if (0 != applicationInfo.flags and ApplicationInfo.FLAG_DEBUGGABLE) {
        DebugWebViewClient(nextWebViewId, webView, nativeProxy, firstLoadCallback)
      } else {
        ProductionWebViewClient(nextWebViewId, webView, nativeProxy, firstLoadCallback)
      }
      nextWebViewId++

      webView.webViewClient = client
      webView.tag = client

      return client
    }

  @SuppressLint("SetJavaScriptEnabled")
  private fun createWebView(intent: Intent, nativeProxy: NativeProxyBase): WebView {

    // Remove the old WebView
    destroyWebView()

    // Create a new WebView
    LayoutInflater.from(this).inflate(R.layout.starshot_webview, rootContainer) as ViewGroup
    val newWebView: WebView = rootContainer.findViewById(R.id.webview)
    webView = newWebView

    val client = createWebViewClient(newWebView, nativeProxy)
    Timber.d("StarshotActivity.createWebView: Created webview ${client.id}")

    // Enable JS for our webview
    val webSettings: WebSettings = newWebView.settings
    webSettings.javaScriptEnabled = true

    // Enable localStorage for the webview
    webSettings.domStorageEnabled = true

    // Enable us to programmatically call play() on video elements outside of an event handler
    webSettings.mediaPlaybackRequiresUserGesture = false

    // Set up debug settings
    if (0 != applicationInfo.flags and ApplicationInfo.FLAG_DEBUGGABLE) {
      setupDebugOptions(webSettings)
    }

    // Set a staging host if set in launch parameters (not allowed for prod builds)
    if (BuildConfig.BUILD_TYPE != "shipping") {
      val starshotHost = intent.getStringExtra("debug-starshot-host")
      if (starshotHost != null) {
        StarshotConfig.setAppHost(starshotHost)
      }
    }

    return newWebView
  }

  private fun destroyWebView() {
    val view = webView
    if (view != null) {
      webView = null
      mediaSession.mediaSessionListener = null

      val client: WebViewClientBase = view.tag as WebViewClientBase

      view.clearFocus()
      Timber.d("StarshotActivity.destroyWebView: Destroying webview ${client.id}")
      client.cleanup()

      view.loadUrl("about:blank")
      rootContainer.removeView(view)

      // Destroy this webview in a short while to let the viewgroup finish removal from the graph
      // Some older versions of Android are throwing an exception internally if we don't do this
      rootContainer.postDelayed(
        {
          view.destroy()
        },
        1000
      )
    }
  }

  private fun loadErrorScreen() {
    Timber.d("StarshotActivity.loadErrorScreen")

    showLoadingScreen()

    val laserArrayNativeProxy = LaserArrayNativeProxy(NoopScriptInvoker())
    val newWebView = createWebView(intent, laserArrayNativeProxy)

    // Register for one click of the retry button
    laserArrayNativeProxy.laserArrayErrorRetryCallback = {
      // Unregister the error click event because sometimes we get it fired twice
      laserArrayNativeProxy.laserArrayErrorRetryCallback = { }
      // Post this to the main thread since callbacks come in on background threads
      rootContainer.post {
        loadStarshot(intent, forceReload = true)
      }
    }

    errorState.show(newWebView)
  }

  private fun handleVideoPlayerStateChanged(state: String) {
    mediaSession.setVideoPlayerState(state)
  }

  private fun handlePlaybackPositionChanged(playerState: String, positionSeconds: Long) {
    mediaSession.setPlaybackPosition(playerState, positionSeconds)
  }

  private fun loadStarshot(intent: Intent, forceReload: Boolean) {
    Timber.d("StarshotActivity.loadStarshot")

    if (webView != null && !forceReload) {
      Timber.i("Skipping webview load in loadStarshot because webview is already loaded")
      return
    }

    showLoadingScreen()

    val invoker = WebViewScriptInvoker()
    val starshotNativeProxy = StarshotNativeProxy(
      this,
      ::handleVideoPlayerStateChanged,
      ::handlePlaybackPositionChanged,
      invoker
    )

    mediaSession.mediaSessionListener = object : IMediaSessionListener {
      override fun onPlay() {
        starshotNativeProxy.play()
      }
      override fun onPause() {
        starshotNativeProxy.pause()
      }
      override fun onStop() {
        starshotNativeProxy.stop()
      }
      override fun onFastForward() {
        starshotNativeProxy.fastForward()
      }
      override fun onRewind() {
        starshotNativeProxy.rewind()
      }
      override fun onSeekTo(pos: Long) {
        val posSeconds = getSecondsFromMs(pos)
        starshotNativeProxy.seekTo(posSeconds)
      }
    }

    val newWebView = createWebView(intent, starshotNativeProxy)
    invoker.webView = newWebView
    mainState.show(newWebView, intent)
  }

  private fun webViewLoaded(webView: WebView) {
    val client: WebViewClientBase = webView.tag as WebViewClientBase
    Timber.d("StarshotActivity.webViewLoaded [${client.id}]")
    crossFadeViews(loadingView, webView, crossFadeDurationMillis)
  }

  private fun setupDebugOptions(webSettings: WebSettings) {
    // Enable debugger
    WebView.setWebContentsDebuggingEnabled(true)

    webSettings.allowFileAccessFromFileURLs = true
    webSettings.allowUniversalAccessFromFileURLs = true
    webSettings.allowContentAccess = true
    webSettings.allowFileAccess = true
    webSettings.databaseEnabled = true
    webSettings.domStorageEnabled = true

    CookieManager.setAcceptFileSchemeCookies(true)

    val cookieManager = CookieManager.getInstance()
    cookieManager.setAcceptCookie(true)
    cookieManager.acceptCookie()
    cookieManager.setAcceptCookie(true)
  }
}

/**
 * Manages the application state where Starshot is being rendered normally in the web view.
 */
class MainState(
  private val context: Context
) {
  fun show(webView: WebView, intent: Intent) {
    updateCookies()

    // Parse deeplink if we were launched with an intent URI
    val baseUri = if (intent.data != null) {
      parseDeeplink(intent.data!!)
    } else {
      StarshotConfig.getAppUrl()
    }

    val builder = baseUri.buildUpon()
      .appendQueryParameter("lnchv", BuildConfig.VERSION_NAME)
      .appendQueryParameter("ipc-version", "android-1")

    // Add any debug url params
    if (0 != context.applicationInfo.flags and ApplicationInfo.FLAG_DEBUGGABLE) {
      // NOTE: Theses parameters are expected to already have been escaped
      val urlParams = intent.getStringExtra("debug-url-params")
      if (urlParams != null) {
        val params = urlParams.split('&')
        for (p in params) {
          if (p.isEmpty()) {
            continue
          }
          val tokens = p.split('=')
          if (tokens.isEmpty()) {
            continue
          }
          val key = URLDecoder.decode(tokens[0], "UTF-8")
          if (key.isEmpty()) {
            continue
          }
          var value = ""
          if (tokens.size > 1) {
            value = URLDecoder.decode(tokens[1], "UTF-8")
          }
          builder.appendQueryParameter(key, value)
        }
      }
    }

    val finalUrl = builder.build()
    webView.loadUrl(finalUrl.toString())
  }

  private fun setCookie(name: String, value: String, secure: Boolean) {
    val secureAttribute = if (secure) {
      " Secure;"
    } else {
      ""
    }

    // Match the behavior of web
    val maxAgeSeconds = "${390 * 24 * 60 * 60}"

    val cookieManager = CookieManager.getInstance()
    cookieManager.setCookie(
      StarshotConfig.getCookieDomain(),
      "$name=$value;$secureAttribute SameSite=None; Max-Age=$maxAgeSeconds"
    )
    cookieManager.setCookie(
      StarshotConfig.getCookieDomain(),
      "${name}_samesite_compat=$value;$secureAttribute Expires=31 Dec 2999 23:59:59 GMT"
    )
  }

  private fun clearCookie(name: String) {
    val cookieManager = CookieManager.getInstance()
    cookieManager.setCookie(
      StarshotConfig.getCookieDomain(),
      "$name=; Max-Age=0"
    )
    cookieManager.setCookie(
      StarshotConfig.getCookieDomain(),
      "${name}_samesite_compat=; Max-Age=0"
    )
  }

  private fun updateCookies() {
    // Update auth-token
    val credentials = getCredentials(context)
    if (credentials.accessToken != "") {
      setCookie("auth-token", credentials.accessToken, true)
    } else {
      clearCookie("auth-token")
    }

    // Update unique_id
    val deviceId = getDeviceId(context)
    setCookie("unique_id", deviceId, false)
  }
}

/**
 * Manages the application state where the offline error page is being displayed.
 */
class ErrorState {
  fun show(webView: WebView) {
    webView.loadUrl("file:///android_asset/laserarray/index.html?mode=native-error")
  }
}
