package tv.twitch.starshot64.app

import android.content.Context
import timber.log.Timber
import tv.twitch.starshot64.analytics.NativeShellLoginTrackingEvent
import tv.twitch.starshot64.analytics.NativeShellLogoutTrackingEvent
import tv.twitch.starshot64.analytics.sendTrackingEvent
import tv.twitch.starshot64.util.TwitchCredentials
import tv.twitch.starshot64.util.clearCredentials
import tv.twitch.starshot64.util.setCredentials

private const val MEDIA_SESSION_WINDOW_PROPERTY_NAME = "window.mediaSessionProxy"

/**
 * Manages calls between native and Starshot running in a webview.
 */
class StarshotNativeProxy(
  private val context: Context,
  private val playerStateChanged: (state: String) -> Unit,
  private val playbackPositionChanged: (playerState: String, positionSeconds: Long) -> Unit,
  javascriptHooks: IInvokeJavascript
) : NativeProxyBase(javascriptHooks) {

  //region Callbacks from Starshot

  @android.webkit.JavascriptInterface
  fun loggedIn(oauthToken: String, refreshToken: String) {
    Timber.d("Login received from web app")

    setCredentials(context, TwitchCredentials(oauthToken, refreshToken))

    sendTrackingEvent(NativeShellLoginTrackingEvent())
  }

  @android.webkit.JavascriptInterface
  fun loggedOut() {
    Timber.d("Logged out received from web app")
    clearCredentials(context)

    sendTrackingEvent(NativeShellLogoutTrackingEvent())
  }

  @android.webkit.JavascriptInterface
  fun videoPlayerStateChanged(state: String) {
    Timber.d("videoPlayerStateChanged received from web app: $state")

    playerStateChanged(state)
  }

  @android.webkit.JavascriptInterface
  fun setPlaybackPosition(playerState: String, positionSeconds: Long) {
    Timber.d("setPlaybackPosition received from web app: $playerState, $positionSeconds")

    playbackPositionChanged(playerState, positionSeconds)
  }

  //endregion

  //region API to call into Starshot

  fun play() {
    javascriptHooks.invoke("if ($MEDIA_SESSION_WINDOW_PROPERTY_NAME) { $MEDIA_SESSION_WINDOW_PROPERTY_NAME.play(); }") // ktlint-disable max-line-length
  }

  fun pause() {
    javascriptHooks.invoke("if ($MEDIA_SESSION_WINDOW_PROPERTY_NAME) { $MEDIA_SESSION_WINDOW_PROPERTY_NAME.pause(); }") // ktlint-disable max-line-length
  }

  fun stop() {
    javascriptHooks.invoke("if ($MEDIA_SESSION_WINDOW_PROPERTY_NAME) { $MEDIA_SESSION_WINDOW_PROPERTY_NAME.stop(); }") // ktlint-disable max-line-length
  }

  fun fastForward() {
    javascriptHooks.invoke("if ($MEDIA_SESSION_WINDOW_PROPERTY_NAME) { $MEDIA_SESSION_WINDOW_PROPERTY_NAME.fastForward(); }") // ktlint-disable max-line-length
  }

  fun rewind() {
    javascriptHooks.invoke("if ($MEDIA_SESSION_WINDOW_PROPERTY_NAME) { $MEDIA_SESSION_WINDOW_PROPERTY_NAME.rewind(); }") // ktlint-disable max-line-length
  }

  fun seekTo(positionSeconds: Long) {
    javascriptHooks.invoke("if ($MEDIA_SESSION_WINDOW_PROPERTY_NAME) { $MEDIA_SESSION_WINDOW_PROPERTY_NAME.seekTo($positionSeconds); }") // ktlint-disable max-line-length
  }

  //endregion
}
