package tv.twitch.starshot64.app

import android.annotation.SuppressLint
import android.graphics.Bitmap
import android.net.http.SslError
import android.webkit.SslErrorHandler
import android.webkit.WebResourceError
import android.webkit.WebResourceRequest
import android.webkit.WebView
import android.webkit.WebViewClient
import timber.log.Timber

const val WINDOW_HOOK_NAME: String = "androidTvNativeHooks"

/**
 * Base class to be used for WebViewClient implementations specific to the application.
 */
abstract class WebViewClientBase(
  val id: Long,
  private val webView: WebView,
  private val nativeAppProxy: NativeProxyBase,
  private val firstPageLoadFinishedCallback: (succeeded: Boolean) -> Unit
) : WebViewClient() {

  private var registered: Boolean = false
  private var firstPageFinished: Boolean = false
  private var pageLoadFailed: Boolean = false

  init {
    registerHooks()
  }

  @SuppressLint("JavascriptInterface")
  private fun registerHooks() {
    if (!registered) {
      webView.addJavascriptInterface(nativeAppProxy, WINDOW_HOOK_NAME)
      registered = true
    }
  }

  private fun unregisterHooks() {
    if (registered) {
      webView.removeJavascriptInterface(WINDOW_HOOK_NAME)
      registered = false
    }
  }

  override fun onPageStarted(view: WebView, url: String, favicon: Bitmap?) {
    super.onPageStarted(view, url, favicon)

    Timber.i("onPageStarted [$id]: ${if (registered) "first" else ""} $url")

    if (webView != view) {
      Timber.i("WebView mismatch")
    }

    // Add access to the native API.  This needs to be set every time the page loads, even when
    // the load happens because of a Next fast refresh.
    unregisterHooks()
    registerHooks()
  }

  override fun onPageFinished(view: WebView, url: String) {
    super.onPageFinished(view, url)

    if (firstPageFinished) {
      Timber.i("onPageFinished [$id]: already finished $url")
      return
    } else {
      Timber.i("onPageFinished [$id]: $url")
    }
    firstPageFinished = true

    // Notify the activity
    firstPageLoadFinishedCallback(!pageLoadFailed)
  }

  override fun onReceivedError(
    view: WebView,
    errorCode: Int,
    description: String,
    failingUrl: String
  ) {
    // Add this override of onReceivedError to catch errors on older versions of Android
    super.onReceivedError(view, errorCode, description, failingUrl)

    if (failingUrl == view.url) {
      handleError()
    }
  }

  override fun onReceivedError(
    view: WebView,
    request: WebResourceRequest,
    error: WebResourceError
  ) {
    // Add this override of onReceivedError to catch errors on newer versions of Android
    super.onReceivedError(view, request, error)

    if (request.isForMainFrame) {
      handleError()
    }
  }

  private fun handleError() {
    if (!firstPageFinished) {
      Timber.i("handleError [$id]: Received error for main frame")
      pageLoadFailed = true
    } else {
      Timber.i("handleError [$id]: Received error for subsequent page")
    }
  }

  fun cleanup() {
    unregisterHooks()
  }
}

/**
 * WebViewClient implementation in debug and QA builds.
 */
class DebugWebViewClient(
  id: Long,
  webView: WebView,
  nativeAppProxy: NativeProxyBase,
  firstPageLoadFinishedCallback: (succeeded: Boolean) -> Unit
) : WebViewClientBase(id, webView, nativeAppProxy, firstPageLoadFinishedCallback) {
  /**
   * When hosting locally we do not have a properly signed cert so this will ignore those errors.
   * This is only to be enabled when the host app is built in debug.
   */
  override fun onReceivedSslError(view: WebView, handler: SslErrorHandler, error: SslError) {
    // Ignore SSL certificate errors so we can connect to developer hosts
    handler.proceed()
  }
}

/**
 * The WebViewClient implementation to be used in a build to be released.
 */
class ProductionWebViewClient(
  id: Long,
  webView: WebView,
  nativeAppProxy: NativeProxyBase,
  firstPageLoadFinishedCallback: (succeeded: Boolean) -> Unit
) : WebViewClientBase(id, webView, nativeAppProxy, firstPageLoadFinishedCallback)
