package tv.twitch.starshot64.app

import android.net.Uri
import kotlin.reflect.KFunction1
import timber.log.Timber

/**
 * Supported intent URI deeplinks:
 *   twitch://home
 *   twitch://browse
 *   twitch://stream/<channel>
 *   twitch://game/<game>
 *   twitch://video/<videoid>
 *   twitch://video/<videoid>?t=<seconds>
 *   twitch://clip/<channel>/<clipid>
 *   twitch://search?q=<query>
 *
 * tt_content and tt_medium will be passed along in the Starshot URL.
 *
 * To test deeplinks on the command line:
 *   adb shell am start -d "twitch://home"
 *   adb shell am start -d "twitch://browse"
 *   adb shell am start -d "twitch://stream/lirik"
 *   adb shell am start -d "twitch://game/minecraft"
 *   adb shell am start -d "twitch://video/898150101"
 *   adb shell am start -d "twitch://video/898150101?t=120"
 *   adb shell am start -d "twitch://clip/fnordromph/DelightfulConfidentSwallowPMSTwin"
 *   adb shell am start -d "twitch://search?q=lirik"
 *
 *   If testing tt_content and tt_medium append the following to the URIs:
 *     ?tt_content=foo\&tt_medium=bar"
 */
fun parseDeeplink(intentUri: Uri): Uri {

  // Invalid protocol, use default
  if (intentUri.scheme != "twitch") {
    Timber.d("encountered unknown deeplink scheme: ${intentUri.scheme}")
    return getStarshotHomeUrl()
  }

  // Parse the give URL
  val parseFuncMap: Map<Int, KFunction1<Uri, Uri?>> = mapOf(
    0 to ::parsePathNoSegments,
    1 to ::parsePathOneSegment,
    2 to ::parsePathTwoSegments
  )

  var parsedUrl: Uri? = if (intentUri.pathSegments.size in parseFuncMap) {
    parseFuncMap[intentUri.pathSegments.size]?.invoke(intentUri)
  } else {
    null
  }

  // Create default URL if parsing failed
  if (parsedUrl == null) {
    Timber.d("unexpected deeplink format, going home: $intentUri")
    parsedUrl = getStarshotHomeUrl()
  }

  // Append tracking parameters
  val builder = parsedUrl.buildUpon()

  val ttContent: String? = intentUri.getQueryParameter("tt_content")
  if (!ttContent.isNullOrEmpty()) {
    builder.appendQueryParameter("tt_content", ttContent)
  }

  val ttMedium: String? = intentUri.getQueryParameter("tt_medium")
  if (!ttMedium.isNullOrEmpty()) {
    builder.appendQueryParameter("tt_medium", ttMedium)
  }

  return builder.build()
}

/**
 * Parses URI such as twitch://home.
 */
private fun parsePathNoSegments(intentUri: Uri): Uri? {
  return when (intentUri.host) {
    "home" -> getStarshotHomeUrl()
    "browse" -> getStarshotBrowseUrl()
    "search" -> {
      val value: String? = intentUri.getQueryParameter("q")
      var query = ""
      if (value != null) {
        query = value
      }
      getStarshotSearchUrl(query)
    }
    else -> {
      null
    }
  }
}

/**
 * Parses URI such as twitch://stream/<channel>.
 */
private fun parsePathOneSegment(intentUri: Uri): Uri? {
  return when (intentUri.host) {
    "stream" -> getStarshotStreamUrl(intentUri.pathSegments[0])
    "game" -> getStarshotGameUrl(intentUri.pathSegments[0])
    "video" -> {
      var time = 0
      val timeString: String? = intentUri.getQueryParameter("t")
      if (timeString != null) {
        val parsed = timeString.toIntOrNull()
        if (parsed != null) {
          time = parsed
        }
      }
      getStarshotVideoUrl(intentUri.pathSegments[0], time)
    }
    else -> {
      null
    }
  }
}

/**
 * Parses URI such as "twitch://clip/fnordromph/DelightfulConfidentSwallowPMSTwin"
 */
private fun parsePathTwoSegments(intentUri: Uri): Uri? {
  return when (intentUri.host) {
    // twitch://clip/<channel>/<clipid>
    "clip" -> getStarshotClipUrl(intentUri.pathSegments[0], intentUri.pathSegments[1])
    else -> {
      null
    }
  }
}

fun getStarshotHomeUrl(): Uri {
  return StarshotConfig.getAppUrl()
}

fun getStarshotBrowseUrl(): Uri {
  return StarshotConfig.getAppUrl()
    .buildUpon()
    .appendPath("directory")
    .build()
}

fun getStarshotStreamUrl(channelName: String): Uri {
  return StarshotConfig.getAppUrl()
    .buildUpon()
    .appendPath(channelName)
    .build()
}

fun getStarshotGameUrl(gameName: String): Uri {
  return StarshotConfig.getAppUrl()
    .buildUpon()
    .appendPath("directory")
    .appendPath("game")
    .appendPath(gameName)
    .build()
}

fun getStarshotVideoUrl(videoId: String, seekOffset: Int = 0): Uri {
  val builder = StarshotConfig.getAppUrl()
    .buildUpon()
    .appendPath("videos")
    .appendPath(videoId)
  if (seekOffset > 0) {
    builder.appendQueryParameter("t", seekOffset.toString())
  }
  return builder.build()
}

fun getStarshotClipUrl(channelName: String, clipId: String): Uri {
  return StarshotConfig.getAppUrl()
    .buildUpon()
    .appendPath(channelName)
    .appendPath("clip").appendPath(clipId)
    .build()
}

fun getStarshotSearchUrl(searchTerm: String): Uri {
  return StarshotConfig.getAppUrl()
    .buildUpon()
    .appendPath("search")
    .appendQueryParameter("term", searchTerm)
    .build()
}
