package tv.twitch.starshot64.media

import android.support.v4.media.session.PlaybackStateCompat
import java.util.Locale

/**
 * Represents a Twitch Player state and provides conversion to the appropriate
 * Java media session state.
 *
 * https://git.xarth.tv/video/player-core/blob/master/platforms/web/js/event/state.ts
 *
 * The state mapping in the legacy TVAPP:
 *
 *   twitch::Player::State::Idle      -> app::IVideoPlayer::VideoPlayerState::Paused    -> PlaybackStateCompat.STATE_PAUSED
 *   twitch::Player::State::Ready     -> app::IVideoPlayer::VideoPlayerState::Ready     -> PlaybackStateCompat.STATE_PAUSED
 *   twitch::Player::State::Buffering -> app::IVideoPlayer::VideoPlayerState::Buffering -> PlaybackStateCompat.STATE_BUFFERING
 *   twitch::Player::State::Playing   -> app::IVideoPlayer::VideoPlayerState::Playing   -> PlaybackStateCompat.STATE_PLAYING
 *   twitch::Player::State::Ended     -> app::IVideoPlayer::VideoPlayerState::Stopped   -> PlaybackStateCompat.STATE_STOPPED
 */
enum class PlayerState(val playerCoreState: String, val mediaSessionPlaybackState: Int) {
  Idle("idle", PlaybackStateCompat.STATE_PAUSED),
  Ready("ready", PlaybackStateCompat.STATE_PAUSED),
  Buffering("buffering", PlaybackStateCompat.STATE_BUFFERING),
  Playing("playing", PlaybackStateCompat.STATE_PLAYING),
  Ended("ended", PlaybackStateCompat.STATE_STOPPED);

  override fun toString(): String {
    return playerCoreState
  }

  companion object {
    fun lookup(value: String): PlayerState? {
      val lower = value.toLowerCase(Locale.ROOT)
      for (type in values()) {
        if (type.playerCoreState == lower) {
          return type
        }
      }
      return null
    }

    fun toMediaSessionString(state: PlayerState): String {
      return toMediaSessionString(state.mediaSessionPlaybackState)
    }

    fun toMediaSessionString(state: Int): String {
      return when (state) {
        PlaybackStateCompat.STATE_NONE -> "STATE_NONE"
        PlaybackStateCompat.STATE_STOPPED -> "STATE_STOPPED"
        PlaybackStateCompat.STATE_PAUSED -> "STATE_PAUSED"
        PlaybackStateCompat.STATE_PLAYING -> "STATE_PLAYING"
        PlaybackStateCompat.STATE_BUFFERING -> "STATE_BUFFERING"
        else -> String.format("Unsupported state: %d", state)
      }
    }
  }
}
