package tv.twitch.starshot64.media

import android.content.Context
import android.support.v4.media.session.MediaSessionCompat
import android.support.v4.media.session.PlaybackStateCompat
import timber.log.Timber
import tv.twitch.starshot64.app.StarshotConfig.Companion.isAndroidTv

private const val MEDIA_SESSION_TAG = "TwitchMediaSession"

/**
 * MediaSession allows the play, pause, stop, rewind fast-forward and seek buttons/voice commands
 * to work with video in our app.
 *
 * Currently, these operations only work on VODs and clips.
 *
 * Test simulate media keys:
 *
 * Play:         adb shell input keyevent 126
 * Pause:        adb shell input keyevent 127
 * Play/Pause:   adb shell input keyevent 85
 * Stop:         adb shell input keyevent 86
 * Rewind:       adb shell input keyevent 89
 * Fast forward: adb shell input keyevent 90
 */
class TwitchMediaSession {
  private val logger = Timber.tag(MEDIA_SESSION_TAG)
  private var mediaSession: MediaSessionCompat? = null
  private var lastKnownPlaybackState = PlaybackStateCompat.STATE_NONE
  private var lastKnownPlaybackPosition = PlaybackStateCompat.PLAYBACK_POSITION_UNKNOWN
  private val supportedPlaybackActions = (
    PlaybackStateCompat.ACTION_PLAY_PAUSE
      or PlaybackStateCompat.ACTION_PLAY
      or PlaybackStateCompat.ACTION_PAUSE
      or PlaybackStateCompat.ACTION_STOP
      or PlaybackStateCompat.ACTION_FAST_FORWARD
      or PlaybackStateCompat.ACTION_REWIND
      or PlaybackStateCompat.ACTION_SEEK_TO
    )

  var mediaSessionListener: IMediaSessionListener? = null

  //region Activity lifecycle hooks

  /**
   * Call this inside Activity.onCreate().
   */
  fun onActivityCreate(context: Context) {
    if (mediaSession != null) {
      logger.e("onCreate: Already created, ignoring")
      return
    }

    logger.d("onCreate() called")

    val session = MediaSessionCompat(context, MEDIA_SESSION_TAG)
    mediaSession = session

    val state = PlaybackStateCompat.Builder()
      .setActions(supportedPlaybackActions)
      .setState(PlaybackStateCompat.STATE_NONE, lastKnownPlaybackPosition, 1.0f)
      .build()
    session.setPlaybackState(state)
    session.setFlags(
      MediaSessionCompat.FLAG_HANDLES_MEDIA_BUTTONS or
        MediaSessionCompat.FLAG_HANDLES_TRANSPORT_CONTROLS
    )

    // Capture the calls from the media system
    session.setCallback(object : MediaSessionCompat.Callback() {
      override fun onPlay() {
        logger.d("MediaSessionCompat.Callback.onPlay()")
        super.onPlay()
        notifyOnPlay()
      }

      override fun onPause() {
        logger.d("MediaSessionCompat.Callback.onPause()")
        super.onPause()
        notifyOnPause()
      }

      override fun onStop() {
        logger.d("MediaSessionCompat.Callback.onStop()")
        super.onStop()
        notifyOnStop()
      }

      override fun onFastForward() {
        logger.d("MediaSessionCompat.Callback.onFastForward()")
        super.onFastForward()
        notifyOnFastForward()
      }

      override fun onRewind() {
        logger.d("MediaSessionCompat.Callback.onRewind()")
        super.onRewind()
        notifyOnRewind()
      }

      override fun onSeekTo(seconds: Long) {
        var position = seconds
        logger.d("MediaSessionCompat.Callback.onSeekTo($position)")

        // Clamp position
        if (position < 0) {
          logger.d("Clamping negative seek position to 0")
          position = 0
        }

        super.onSeekTo(position)
        notifyOnSeekTo(position)
      }
    })
  }

  /**
   * Call this inside Activity.onStart().
   */
  fun onActivityStart() {
    // Placeholder for now for symmetry in the API
  }

  /**
   * Call this inside Activity.onPause().
   */
  fun onActivityPause() {
    val session = mediaSession
    if (session == null) {
      logger.d("Error: trying to deactivate media session when none exists!")
      return
    }

    logger.d(
      "TwitchMediaSession.onPause() called, last known state: ${PlayerState.toMediaSessionString(lastKnownPlaybackState)}" // ktlint-disable max-line-length
    )

    // On Android TV the assistant actually pauses our activity when it is triggered, and we need
    // our media session to stay active so it can receive voice commands from the assistant
    if (isAndroidTv()) {
      updateState(PlaybackStateCompat.STATE_PAUSED, lastKnownPlaybackPosition, false)
      session.isActive = false
    }
  }

  /**
   * Call this inside Activity.onResume().
   */
  fun onActivityResume() {
    val session = mediaSession
    if (session == null) {
      logger.d("Error: trying to activate media session when none exists!")
      return
    }

    logger.d(
      "TwitchMediaSession.onResume() called, last known state: ${PlayerState.toMediaSessionString(lastKnownPlaybackState)}" // ktlint-disable max-line-length
    )

    updateState(lastKnownPlaybackState, lastKnownPlaybackPosition, false)
    session.isActive = true
  }

  /**
   * Call this inside Activity.onStop().
   */
  fun onActivityStop() {
    val session = mediaSession
    if (session == null) {
      logger.d("Error: trying to deactivate media session when none exists!")
      return
    }

    logger.d(
      "TwitchMediaSession.onStop() called, last known state: ${PlayerState.toMediaSessionString(lastKnownPlaybackState)}" // ktlint-disable max-line-length
    )

    updateState(PlaybackStateCompat.STATE_PAUSED, lastKnownPlaybackPosition, false)
    session.isActive = false
  }

  /**
   * Call this inside Activity.onDestroy().
   */
  fun onActivityDestroy() {
    logger.d("TwitchMediaSession.destroy() called")

    if (mediaSession == null) {
      logger.d("Error: trying to deactivate media session when none exists!")
      return
    }

    mediaSession!!.release()
    mediaSession = null
  }

  //endregion

  //region Sync state with real player

  /**
   * To be called when the state of the application player changes.
   * @param playerState The state string from the web application player.
   */
  fun setVideoPlayerState(playerState: String) {
    if (mediaSession == null) {
      logger.w("onStateChanged: trying to update media session state when no media session exists!") // ktlint-disable max-line-length
      return
    }

    val state = PlayerState.lookup(playerState)
    if (state == null) {
      logger.w(
        "TwitchMediaSession.onStateChanged: Unexpected state value received: $playerState"
      )
      return
    }

    logger.d("TwitchMediaSession.onStateChanged() called, state: $state")

    updateState(state.mediaSessionPlaybackState, lastKnownPlaybackPosition, true)
  }

  /**
   * To be called periodically by the video player to inform of the current position.
   * This is only relevant during VODs and clips.
   */
  fun setPlaybackPosition(playerState: String, playbackPosition: Long) {
    if (mediaSession == null) {
      logger.w("onPlaybackPositionChanged: trying to update media session state when no media session exists!") // ktlint-disable max-line-length
      return
    }

    val state = PlayerState.lookup(playerState)
    if (state == null) {
      logger.w(
        "TwitchMediaSession.onPlaybackPositionChanged: Unexpected state value received: $playerState" // ktlint-disable max-line-length
      )
      return
    }

    logger.v(
      "TwitchMediaSession.onPlaybackPositionChanged() called, state: $state, position: $playbackPosition" // ktlint-disable max-line-length
    )

    lastKnownPlaybackPosition = playbackPosition
    updateState(state.mediaSessionPlaybackState, playbackPosition, true)
  }

  /**
   * Updates the system state based on various sources of truth and updates the system media
   * session.
   */
  private fun updateState(playbackState: Int, playbackPosition: Long, cacheNewState: Boolean) {
    try {
      val session = mediaSession
      if (session == null) {
        logger.w("TwitchMediaSession.updateState(): trying to update media session state when no media session exists!") // ktlint-disable max-line-length
        return
      }

      val newState = PlaybackStateCompat.Builder()
        .setActions(supportedPlaybackActions)
        .setState(playbackState, playbackPosition, 1.0f)
        .build()

      logger.d(
        "TwitchMediaSession.updateState(): state: ${PlayerState.toMediaSessionString(playbackState)}, position: $playbackPosition" // ktlint-disable max-line-length
      )

      session.setPlaybackState(newState)

      if (cacheNewState) {
        lastKnownPlaybackState = playbackState
      }
    } catch (e: Exception) {
      logger.e("Exception when trying to update media session state: $e")
    }
  }

  //endregion

  //region Notification helpers

  private fun notifyOnPlay() {
    mediaSessionListener?.onPlay()
  }

  private fun notifyOnPause() {
    mediaSessionListener?.onPause()
  }

  private fun notifyOnStop() {
    mediaSessionListener?.onStop()
  }

  private fun notifyOnFastForward() {
    mediaSessionListener?.onFastForward()
  }

  private fun notifyOnRewind() {
    mediaSessionListener?.onRewind()
  }

  private fun notifyOnSeekTo(pos: Long) {
    mediaSessionListener?.onSeekTo(pos)
  }

  // endregion
}
