package tv.twitch.starshot64.recommendations

import android.content.BroadcastReceiver
import android.content.Context
import android.content.Intent
import androidx.tvprovider.media.tv.TvContractCompat
import timber.log.Timber
import tv.twitch.starshot64.analytics.initializeTracking

/**
 * Initializes channels and programs when the device boots up.  It also listens for
 * locale change events, and for the user removing items from the watch next
 * row.
 *
 * NOTE: The app needs to be launched once in order for the service to be registered.  Also,
 * if you Force Quit the app it will terminate the recommendations service.
 */
class RecommendationsBroadcastReceiver : BroadcastReceiver() {
  companion object {
    private val logger = Timber.tag("RecommBroadcastReceiver")
  }

  override fun onReceive(context: Context, intent: Intent?) {
    logger.d("onReceive(): %s", intent)

    if (intent == null || intent.action.isNullOrEmpty()) {
      return
    }

    // Initialize the tracking subsystem
    initializeTracking(context)

    val recommendationsStrategy = getRecommendationsStrategy()

    val action = intent.action!!
    when {
      action.endsWith(Intent.ACTION_BOOT_COMPLETED) ->
        recommendationsStrategy.handleBootCompleted(context)
      action.endsWith(Intent.ACTION_LOCALE_CHANGED) ->
        recommendationsStrategy.handleLocaleChanged(context)
      action.endsWith(TvContractCompat.ACTION_INITIALIZE_PROGRAMS) ->
        recommendationsStrategy.handleInitializePrograms(context)
      action.endsWith(TvContractCompat.ACTION_WATCH_NEXT_PROGRAM_BROWSABLE_DISABLED) -> {
        val programId =
          intent.getLongExtra(TvContractCompat.ACTION_WATCH_NEXT_PROGRAM_BROWSABLE_DISABLED, -1)
        recommendationsStrategy.handleRemoveWatchNextProgram(context, programId)
      }
    }
  }
}
