package tv.twitch.starshot64.recommendations.channel

import android.content.Context
import android.net.Uri
import androidx.tvprovider.media.tv.TvContractCompat
import timber.log.Timber
import tv.twitch.fragment.GameFragment
import tv.twitch.fragment.StreamFragment
import tv.twitch.fragment.VideoFragment
import tv.twitch.starshot64.R
import tv.twitch.starshot64.recommendations.IRecommendationsStrategy
import tv.twitch.starshot64.recommendations.createGameUri
import tv.twitch.starshot64.recommendations.createHomeUri
import tv.twitch.starshot64.recommendations.createStreamUri
import tv.twitch.starshot64.recommendations.createVideoUri
import tv.twitch.starshot64.util.currentEpochInSeconds
import tv.twitch.starshot64.util.parseTwitchISO8601
import tv.twitch.starshot64.util.unlocalizedFormat

private const val RECOMMENDED_STREAMS_TT_CONTENT = "launcher_recommended_streams"
private const val RECOMMENDED_GAMES_TT_CONTENT = "launcher_recommended_games"
private const val WATCH_NEXT_TT_CONTENT = "launcher_watch_next"

class ChannelRecommendationsStrategy :
  IRecommendationsStrategy {

  companion object {
    private val logger = Timber.tag("ChannelRecommStrategy")
  }

  override fun handleActivityStarted(context: Context) {
    logger.d("handleActivityStarted")
    ChannelRecommendationsWorker.schedulePeriodicWork(context)
  }

  override fun handleBootCompleted(context: Context) {
    logger.d("handleBootCompleted")
    ChannelRecommendationsWorker.schedulePeriodicWork(context)
  }

  override fun handleLocaleChanged(context: Context) {
    logger.d("handleLocaleChanged")
    ChannelRecommendationsWorker.schedulePeriodicWork(context)
  }

  override fun handleInitializePrograms(context: Context) {
    logger.d("handleInitializePrograms")
    ChannelRecommendationsWorker.schedulePeriodicWork(context)
  }

  override fun updateRecommendedStreams(context: Context, streams: Iterable<StreamFragment>) {
    logger.d("updateRecommendedStreams")

    // Create the channel if it didn't exist
    val channelId = findOrCreateChannel(
      context,
      context.getString(R.string.recommended_streams),
      createHomeUri(RECOMMENDED_STREAMS_TT_CONTENT).toString(),
      R.drawable.recommendation_row_glitch
    )

    if (channelId != null) {
      // Make recommended streams the default channel
      TvContractCompat.requestChannelBrowsable(context, channelId)

      // Remove previous programs
      removeProgramsForChannel(context, channelId)

      // Add new programs
      streams.forEach { stream ->
        // Can't make a deeplink if we don't have login
        if (stream.broadcaster?.login == null) {
          return
        }

        val viewerCount = stream.viewersCount ?: 0
        val formattedDescription = if (stream.game?.displayName != null) {
          context.resources.getQuantityString(
            R.plurals.playing_game_for_viewers,
            viewerCount,
            unlocalizedFormat(viewerCount.toLong()),
            stream.game.displayName
          )
        } else {
          context.resources.getQuantityString(
            R.plurals.playing_for_viewers,
            viewerCount,
            unlocalizedFormat(viewerCount.toLong())
          )
        }
        val program = buildPreviewProgram(
          channelId,
          stream.broadcaster.displayName,
          formattedDescription,
          Uri.parse(stream.previewImageURL ?: ""),
          AspectRatio.ASPECT_16_9,
          createStreamUri(stream.broadcaster.login, RECOMMENDED_STREAMS_TT_CONTENT),
          isLiveStream = true
        )
        val programUri = context.contentResolver.insert(
          TvContractCompat.PreviewPrograms.CONTENT_URI, program.toContentValues()
        )
        if (programUri != null) {
          logger.d("Inserted stream program: %s", programUri)
        } else {
          logger.d(
            "Failed to insert program for stream: %s",
            stream.broadcaster.displayName
          )
        }
      }
    } else {
      logger.e("Failed to find or create recommended streams channel!")
    }
  }

  override fun updateRecommendedGames(context: Context, games: Iterable<GameFragment>) {
    // Create the channel if it didn't exist
    val channelId = findOrCreateChannel(
      context,
      context.getString(R.string.recommended_games),
      createHomeUri(RECOMMENDED_GAMES_TT_CONTENT).toString(),
      R.drawable.recommendation_row_glitch
    )

    if (channelId != null) {
      // Remove previous programs
      removeProgramsForChannel(context, channelId)

      // Add new programs
      games.forEach { game ->
        // Can't make a deeplink if we don't have login
        val viewerCount = game.viewersCount ?: 0
        val formattedDescription =
          context.resources.getQuantityString(
            R.plurals.viewers,
            viewerCount,
            unlocalizedFormat(viewerCount.toLong())
          )

        val program = buildPreviewProgram(
          channelId,
          game.displayName,
          formattedDescription,
          Uri.parse(game.boxArtURL ?: ""),
          AspectRatio.ASPECT_BOX_ART,
          createGameUri(game.name, RECOMMENDED_GAMES_TT_CONTENT),
          isLiveStream = false
        )
        val programUri = context.contentResolver.insert(
          TvContractCompat.PreviewPrograms.CONTENT_URI, program.toContentValues()
        )
        if (programUri != null) {
          logger.d("Inserted game program: %s", programUri)
        } else {
          logger.d("Failed to insert program for game: %s", game.name)
        }
      }
    } else {
      logger.e("Failed to find or create recommended games channel!")
    }
  }

  override fun updateWatchNextVideos(context: Context, videos: Iterable<VideoFragment>) {
    val watchNextExpirationSeconds = 30L * 24L * 60L * 60L // 30 days

    // Remove previous programs
    deleteAllWatchNextPrograms(context)

    // Filter programs that don't meet our criteria for showing in the watch next row
    val filteredVideos = videos.filter { video ->
      // Can't schedule a vod that's missing these fields
      video.self?.viewingHistory?.position != null && video.lengthSeconds != null &&
        video.creator != null && video.title != null
    }.filterNot { video ->
      // Hide videos that are less than 5% played or more than 95% played
      val playbackRatio =
        video.self!!.viewingHistory!!.position!!.toFloat() / video.lengthSeconds!!.toFloat()
      playbackRatio < 0.05 || playbackRatio > 0.95
    }.filterNot { video ->
      // Hide videos older than a certain date
      val updatedDate = parseTwitchISO8601(video.self!!.viewingHistory!!.updatedAt!!.toString())
      // Hide videos more than 30 days old
      val now = currentEpochInSeconds()
      now - updatedDate.time > watchNextExpirationSeconds
    }.filterNot { video ->
      // Don't show if the user hid it via the Android UI
      isProgramHiddenByUser(context, video.id)
    }

    // Schedule programs
    filteredVideos.forEach { video ->
      val program = createWatchNextProgram(
        video.id,
        video.creator!!.displayName,
        video.title!!,
        Uri.parse(video.previewThumbnailURL),
        video.lengthSeconds!!,
        video.self!!.viewingHistory!!.position!!,
        parseTwitchISO8601(video.self.viewingHistory!!.updatedAt!!.toString()),
        createVideoUri(video.id, WATCH_NEXT_TT_CONTENT)
      )
      if (program != null) {
        val programUri = context.contentResolver
          .insert(TvContractCompat.WatchNextPrograms.CONTENT_URI, program.toContentValues())

        if (programUri != null) {
          logger.d("Inserted video program: %s", programUri)
        } else {
          logger.d("Failed to insert program for video: %s", video.id)
        }
      }
    }
  }

  override fun handleRemoveWatchNextProgram(context: Context, programId: Long) {
    hideWatchNextProgram(context, programId)
  }
}
