package tv.twitch.starshot64.recommendations.channel

import android.annotation.TargetApi
import android.content.Context
import android.os.Build
import androidx.work.Constraints
import androidx.work.CoroutineWorker
import androidx.work.ExistingPeriodicWorkPolicy
import androidx.work.NetworkType
import androidx.work.OneTimeWorkRequestBuilder
import androidx.work.PeriodicWorkRequestBuilder
import androidx.work.WorkManager
import androidx.work.WorkerParameters
import com.apollographql.apollo.api.toInput
import com.apollographql.apollo.coroutines.await
import java.util.UUID
import java.util.concurrent.TimeUnit
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.withContext
import timber.log.Timber
import tv.twitch.EmpStarshotAndroidRecommendationsQuery
import tv.twitch.fragment.GameFragment
import tv.twitch.fragment.StreamFragment
import tv.twitch.starshot64.net.graphql.newApolloClient
import tv.twitch.starshot64.recommendations.getRecommendationsStrategy
import tv.twitch.starshot64.util.getLanguageCode
import tv.twitch.starshot64.util.isSleepMode
import tv.twitch.starshot64.util.takeFirstNotNullN
import tv.twitch.type.RecommendationsContext

@TargetApi(Build.VERSION_CODES.O)
class ChannelRecommendationsWorker(context: Context, params: WorkerParameters) :
  CoroutineWorker(context, params) {

  companion object {
    private const val RECS_PER_ROW = 10
    private const val WORK_NAME = "SyncChannelsWork"
    private val logger = Timber.tag("ChannelRecommWorker")

    fun scheduleImmediateWork(context: Context) {
      logger.d("scheduling immediate channels work")

      val constraints = Constraints.Builder().setRequiredNetworkType(NetworkType.CONNECTED).build()

      val workRequest =
        OneTimeWorkRequestBuilder<ChannelRecommendationsWorker>().setConstraints(constraints)
          .build()

      WorkManager.getInstance(context).enqueue(workRequest)
    }

    fun schedulePeriodicWork(context: Context) {
      logger.d("scheduling periodic channels work")

      val constraints = Constraints.Builder().setRequiredNetworkType(NetworkType.CONNECTED).build()
      val workRequest =
        PeriodicWorkRequestBuilder<ChannelRecommendationsWorker>(
          15,
          TimeUnit.MINUTES
        ).setConstraints(
          constraints
        ).build()

      WorkManager.getInstance(context).enqueueUniquePeriodicWork(
        WORK_NAME,
        ExistingPeriodicWorkPolicy.KEEP,
        workRequest
      )
    }
  }

  override suspend fun doWork(): Result = withContext(Dispatchers.IO) {
    logger.d("starting work")

    if (isSleepMode(applicationContext)) {
      // Avoid unnecessary work and load on the backend if the device is asleep
      logger.d("Device is asleep, skipping recommendations update")
      Result.success()
    } else {
      deleteLegacyChannels()

      val apollo = newApolloClient(applicationContext)

      val queryResult: kotlin.Result<EmpStarshotAndroidRecommendationsQuery.Data?> =
        try {
          val recsPerRow = 10
          val recsLocation = "TV_APPS"
          val recsPlatform = "android"
          val recsQuery = EmpStarshotAndroidRecommendationsQuery(
            recsPerRow.toInput(),
            getLanguageCode(),
            UUID.randomUUID().toString(),
            recsLocation,
            RecommendationsContext(platform = recsPlatform.toInput())
          )
          kotlin.Result.success(apollo.query(recsQuery).await().data)
        } catch (e: Exception) {
          kotlin.Result.failure(e)
        }

      if (queryResult.isSuccess && queryResult.getOrNull() != null) {
        logger.d("got some recs!")

        val result = queryResult.getOrNull()!!
        val recsStrategy = getRecommendationsStrategy()

        // Create stream recommendations
        val streamSources = ArrayList<Iterable<StreamFragment>?>()
        streamSources += result.currentUser?.followedLiveUsers?.edges?.mapNotNull { edge ->
          edge?.node?.stream?.fragments?.streamFragment
        }
        streamSources += result.recommendedStreams?.edges?.mapNotNull { edge ->
          edge.node?.fragments?.streamFragment
        }
        streamSources += result.featuredStreams?.mapNotNull { edge ->
          edge?.stream?.fragments?.streamFragment
        }

        val streamRecs = takeFirstNotNullN(RECS_PER_ROW, streamSources)
        streamRecs.forEach { stream ->
          logger.d("stream: ${stream.broadcaster!!.displayName}, viewers: ${stream.viewersCount}")
        }
        recsStrategy.updateRecommendedStreams(applicationContext, streamRecs)

        // Create game recommendations
        val gameSources = ArrayList<Iterable<GameFragment>?>()
        gameSources += result.recommendedGames?.edges?.mapNotNull { edge ->
          edge.node?.fragments?.gameFragment
        }
        gameSources += result.games?.edges?.mapNotNull { edge ->
          edge.node?.fragments?.gameFragment
        }

        val gameRecs = takeFirstNotNullN(RECS_PER_ROW, gameSources)
        gameRecs.forEach { game ->
          logger.d("game: ${game.displayName}, viewers: ${game.viewersCount}")
        }
        recsStrategy.updateRecommendedGames(applicationContext, gameRecs)

        // Create watch next recommendations
        val videos = result.currentUser?.viewedVideos?.edges?.mapNotNull { edge ->
          edge?.node?.fragments?.videoFragment
        }

        if (videos != null && videos.isNotEmpty()) {
          videos.forEach { video ->
            logger.d("video: ${video.id}, user: ${video.creator?.displayName}, updatedAt: ${video.self?.viewingHistory?.updatedAt}") // ktlint-disable max-line-length
          }
          recsStrategy.updateWatchNextVideos(applicationContext, videos)
        } else {
          logger.e("Did not receive any watch next videos, row might not show up")
        }

        Result.success()
      } else {
        logger.e("failed to fetch recs")
        queryResult.exceptionOrNull()?.printStackTrace()
        Result.failure()
      }
    }
  }

  /**
   * Remove the channels that the legacy TVAPP defined so we can replace them.
   */
  private fun deleteLegacyChannels() {
    deleteChannel(applicationContext, "twitch://home?referrer=launcher_recommended_recommended")
    deleteChannel(applicationContext, "twitch://home?referrer=launcher_recommended_followed_games")
  }
}
