package tv.twitch.starshot64.recommendations.channel

import android.annotation.TargetApi
import android.content.ContentUris
import android.content.Context
import android.content.Intent
import android.media.tv.TvContract
import android.net.Uri
import android.os.Build
import androidx.tvprovider.media.tv.Channel
import androidx.tvprovider.media.tv.ChannelLogoUtils
import androidx.tvprovider.media.tv.PreviewProgram
import androidx.tvprovider.media.tv.TvContractCompat
import timber.log.Timber
import tv.twitch.starshot64.app.StarshotActivity
import tv.twitch.starshot64.util.convertResourceToBitmap

typealias AndroidChannelId = Long
typealias AndroidProgramId = Long

@TargetApi(Build.VERSION_CODES.O)
private val CHANNELS_PROJECTION = arrayOf(
  TvContractCompat.Channels._ID,
  TvContract.Channels.COLUMN_APP_LINK_INTENT_URI,
  TvContractCompat.Channels.COLUMN_BROWSABLE
)

@TargetApi(Build.VERSION_CODES.O)
private val PROGRAMS_PROJECTION = arrayOf(
  TvContractCompat.Programs._ID,
  TvContract.Programs.COLUMN_CHANNEL_ID
)

/**
 * Return a list of all program IDs that currently exist within the channel for the provided channel ID
 */
@TargetApi(Build.VERSION_CODES.O)
fun findProgramsForChannel(
  context: Context,
  channelId: AndroidChannelId
): List<AndroidProgramId> {
  val result = ArrayList<AndroidProgramId>()
  try {
    context.contentResolver.query(
      TvContractCompat.buildPreviewProgramsUriForChannel(channelId),
      PROGRAMS_PROJECTION,
      null,
      null,
      null
    ).use { cursor ->
      if (cursor != null && cursor.moveToFirst()) {
        do {
          val programId = cursor.getLong(PROGRAMS_PROJECTION.indexOf(TvContractCompat.Programs._ID))
          result.add(programId)
        } while (cursor.moveToNext())
      }
    }
  } catch (ex: Exception) {
    Timber.e(ex, "Unexpected error building program list")
    ex.printStackTrace()
  }
  return result
}

/**
 * Given a deeplink, look up a channel if a channel has been created with this link
 */
@TargetApi(Build.VERSION_CODES.O)
private fun findChannelByDeeplink(
  context: Context,
  deeplink: String
): Channel? {
  try {
    context.contentResolver.query(
      TvContractCompat.Channels.CONTENT_URI,
      CHANNELS_PROJECTION,
      null,
      null,
      null
    ).use { cursor ->
      // Check if our subscription has been added to the launcher channels before
      if (cursor != null && cursor.moveToFirst()) {
        do {
          val channel = Channel.fromCursor(cursor)
          if (deeplink == channel.appLinkIntentUri.toString()) {
            return channel
          }
        } while (cursor.moveToNext())
      }
    }
  } catch (ex: java.lang.Exception) {
    Timber.e(ex, "Unexpected error looking up channel by deeplink")
    ex.printStackTrace()
  }
  return null
}

/**
 * Removes the given channel from the Android home screen.
 */
fun deleteChannel(context: Context, deeplink: String) {
  val channel = findChannelByDeeplink(context, deeplink)
  if (channel != null) {
    val uri = TvContractCompat.buildChannelUri(channel.id)
    context.contentResolver.delete(uri, null, null)
  }
}

/**
 * Given a deeplink, look up the ID of the channel with that link, if one exists
 */
@TargetApi(Build.VERSION_CODES.O)
private fun findChannelIdByDeeplink(context: Context, deeplink: String): AndroidChannelId? {
  return findChannelByDeeplink(context, deeplink)?.id
}

fun findOrCreateChannel(
  context: Context,
  name: String,
  deeplinkUriString: String,
  logoResourceId: Int
): AndroidChannelId? {
  val channel = findChannelByDeeplink(context, deeplinkUriString)
  val channelId: AndroidChannelId?

  // Doesn't exist yet, create it
  if (channel == null) {
    val appLinkIntentUri = Uri.parse(deeplinkUriString)
    val homePageIntent = Intent(context, StarshotActivity::class.java)
    homePageIntent.data = appLinkIntentUri
    val builder = Channel.Builder()
      .setType(TvContractCompat.Channels.TYPE_PREVIEW)
      .setDisplayName(name)
      .setAppLinkIntentUri(appLinkIntentUri)
    Timber.v("Creating channel: $name")
    val channelUrl = context.contentResolver.insert(
      TvContractCompat.Channels.CONTENT_URI,
      builder.build().toContentValues()
    ) ?: return null

    Timber.v("channel insert at $channelUrl")
    val parsedChannelId = ContentUris.parseId(channelUrl)
    Timber.v("channel id $parsedChannelId")
    val bitmap = convertResourceToBitmap(context, logoResourceId)
    ChannelLogoUtils.storeChannelLogo(context, parsedChannelId, bitmap)
    channelId = parsedChannelId
  } else {
    channelId = channel.id
    val appLinkIntentUri = Uri.parse(deeplinkUriString)
    val builder = Channel.Builder()
      .setType(TvContractCompat.Channels.TYPE_PREVIEW)
      .setDisplayName(name)
      .setDescription("") // TODO: determine if this is even needed
      .setAppLinkIntentUri(appLinkIntentUri)
    context.contentResolver.update(
      TvContractCompat.buildChannelUri(channel.id),
      builder.build().toContentValues(),
      null,
      null
    )
  }

  return channelId
}

/**
 * Given a channel id, remove all programs from that channel if it exists
 */
fun removeProgramsForChannel(context: Context, channelId: AndroidChannelId) {
  val existingPrograms = findProgramsForChannel(context, channelId)
  existingPrograms.forEach { programId ->
    context.contentResolver.delete(TvContractCompat.buildPreviewProgramUri(programId), null, null)
  }
}

enum class AspectRatio(val tvProviderAspectRatio: Int) {
  ASPECT_BOX_ART(TvContractCompat.PreviewPrograms.ASPECT_RATIO_MOVIE_POSTER),
  ASPECT_16_9(TvContractCompat.PreviewPrograms.ASPECT_RATIO_16_9),
}

/**
 * Builds a program that can be added to a channel
 */
fun buildPreviewProgram(
  channelId: AndroidChannelId,
  title: String,
  description: String,
  previewImageUri: Uri,
  aspectRatio: AspectRatio,
  intentUri: Uri,
  isLiveStream: Boolean
): PreviewProgram {
  val builder = PreviewProgram.Builder()
  builder.setChannelId(channelId)
    .setType(TvContractCompat.PreviewProgramColumns.TYPE_CLIP)
    .setTitle(title)
    .setDescription(description)
    .setPosterArtUri(previewImageUri)
    .setPosterArtAspectRatio(aspectRatio.tvProviderAspectRatio)
    .setIntentUri(intentUri)
    .setLive(isLiveStream)
  return builder.build()
}
