package tv.twitch.starshot64.recommendations.channel

import android.annotation.TargetApi
import android.content.Context
import android.net.Uri
import android.os.Build
import android.provider.BaseColumns
import androidx.tvprovider.media.tv.TvContractCompat
import androidx.tvprovider.media.tv.WatchNextProgram
import java.util.Date
import timber.log.Timber

private const val COLUMN_WATCH_NEXT_ID_INDEX = 0
private const val COLUMN_WATCH_NEXT_INTERNAL_PROVIDER_ID_INDEX = 1
private val WATCH_NEXT_MAP_PROJECTION = arrayOf(
  BaseColumns._ID,
  TvContractCompat.WatchNextPrograms.COLUMN_INTERNAL_PROVIDER_ID,
  TvContractCompat.WatchNextPrograms.COLUMN_BROWSABLE
)
// Keep the same name as the old Android app for compatibility
private const val HIDDEN_PREFS_KEY =
  "tv.twitch.android.recommendations.api26.hidewatchnext"

@TargetApi(Build.VERSION_CODES.O)
data class QueryProgramResult(val program: WatchNextProgram, val vodId: String, val programId: Long)

@TargetApi(Build.VERSION_CODES.O)
fun createWatchNextProgram(
  videoId: String,
  userName: String,
  broadcastTitle: String,
  previewImageUri: Uri,
  durationSeconds: Int,
  startTimeSeconds: Int,
  updatedAt: Date,
  intentUri: Uri
): WatchNextProgram? {
  return try {
    WatchNextProgram.Builder()
      .setWatchNextType(TvContractCompat.WatchNextPrograms.WATCH_NEXT_TYPE_CONTINUE)
      .setInternalProviderId(videoId)
      .setType(TvContractCompat.PreviewProgramColumns.TYPE_CLIP)
      .setTitle("$userName - $broadcastTitle")
      .setDescription("$userName - $broadcastTitle")
      .setPosterArtUri(previewImageUri)
      .setPosterArtAspectRatio(TvContractCompat.PreviewPrograms.ASPECT_RATIO_16_9)
      .setDurationMillis(durationSeconds * 1000)
      .setLastPlaybackPositionMillis(startTimeSeconds * 1000)
      .setLastEngagementTimeUtcMillis(updatedAt.time)
      .setIntentUri(intentUri)
      .build()
  } catch (ex: Exception) {
    Timber.e(ex, "Unexpected error in createWatchNextProgram")
    null
  }
}

@TargetApi(Build.VERSION_CODES.O)
fun updateWatchNextProgram(
  program: WatchNextProgram,
  previewImageUri: Uri,
  durationSeconds: Int,
  startTimeSeconds: Int,
  updatedAt: Date,
  intentUri: Uri
): WatchNextProgram? {
  try {
    return WatchNextProgram.Builder(program)
      .setPosterArtUri(previewImageUri)
      .setPosterArtAspectRatio(TvContractCompat.PreviewPrograms.ASPECT_RATIO_16_9)
      .setDurationMillis(durationSeconds * 1000)
      .setLastPlaybackPositionMillis(startTimeSeconds * 1000)
      .setLastEngagementTimeUtcMillis(updatedAt.getTime())
      .setIntentUri(intentUri)
      .build()
  } catch (ex: Exception) {
    Timber.e(ex, "Unexpected error in updateWatchNextProgram")
  }
  return null
}

@TargetApi(Build.VERSION_CODES.O)
fun lookupExistingWatchNextProgram(context: Context, videoId: String): QueryProgramResult? {
  try {
    context.contentResolver.query(
      TvContractCompat.WatchNextPrograms.CONTENT_URI,
      WATCH_NEXT_MAP_PROJECTION,
      null,
      null,
      null
    ).use { cursor ->
      if (cursor != null && cursor.moveToFirst()) {
        do {
          val internalId = cursor.getString(COLUMN_WATCH_NEXT_INTERNAL_PROVIDER_ID_INDEX)
          val programId = cursor.getLong(COLUMN_WATCH_NEXT_ID_INDEX)
          if (videoId == internalId) {
            return QueryProgramResult(WatchNextProgram.fromCursor(cursor), internalId, programId)
          }
        } while (cursor.moveToNext())
      }
    }
  } catch (ex: Exception) {
    Timber.e(ex, "Unexpected error in lookupExistingWatchNextProgram")
  }
  return null
}

@TargetApi(Build.VERSION_CODES.O)
fun hideWatchNextProgram(context: Context, watchNextId: Long) {
  try {
    // Loop over all programs until we find the one the user removed from the watch next row, then
    // we get the Twitch ID associated with the video and cache it so we don't add it the next time
    // the service runs.
    context.contentResolver.query(
      TvContractCompat.WatchNextPrograms.CONTENT_URI,
      WATCH_NEXT_MAP_PROJECTION,
      null,
      null,
      null
    ).use { cursor ->
      if (cursor != null && cursor.moveToFirst()) {
        do {
          val programId = cursor.getLong(COLUMN_WATCH_NEXT_ID_INDEX)
          if (programId == watchNextId) {
            val videoId = cursor.getString(COLUMN_WATCH_NEXT_INTERNAL_PROVIDER_ID_INDEX)
            setProgramHiddenByUser(context, videoId)
          }
        } while (cursor.moveToNext())
      }
    }
  } catch (ex: java.lang.Exception) {
    Timber.e(ex, "Unexpected error in hideWatchNextProgram")
  }
}

@TargetApi(Build.VERSION_CODES.O)
fun deleteWatchNextProgram(context: Context, watchNextId: Long) {
  try {
    context.contentResolver
      .delete(TvContractCompat.buildWatchNextProgramUri(watchNextId), null, null)
  } catch (ex: Exception) {
    Timber.e(ex, "Unexpected error in deleteWatchNextProgram")
  }
}

@TargetApi(Build.VERSION_CODES.O)
fun deleteAllWatchNextPrograms(context: Context) {
  deleteAllWatchNextPrograms(context, HashSet())
}

@TargetApi(Build.VERSION_CODES.O)
fun deleteAllWatchNextPrograms(context: Context, excludeIds: Set<String?>) {
  try { // Loop over all programs and gather their IDs
    val idsToDelete = ArrayList<Long>()
    context.contentResolver.query(
      TvContractCompat.WatchNextPrograms.CONTENT_URI,
      WATCH_NEXT_MAP_PROJECTION,
      null,
      null,
      null
    ).use { cursor ->
      if (cursor != null && cursor.moveToFirst()) {
        do {
          val twitchVideoId = cursor.getString(COLUMN_WATCH_NEXT_INTERNAL_PROVIDER_ID_INDEX)
          // Skip deleting if its in the exclude list
          if (!excludeIds.contains(twitchVideoId)) {
            idsToDelete.add(cursor.getLong(COLUMN_WATCH_NEXT_ID_INDEX))
          }
        } while (cursor.moveToNext())
      }
    }
    // Delete each ID (doing this in a separate loop in case mutating while iterating over the cursor
    // causes some weird behavior, and the size of our collection is only a few elements
    for (programId in idsToDelete) {
      deleteWatchNextProgram(context, programId)
    }
  } catch (ex: Exception) {
    Timber.e(ex, "Unexpected error in deleteAllWatchNextPrograms")
  }
}

@TargetApi(Build.VERSION_CODES.O)
fun setProgramHiddenByUser(context: Context, videoId: String) {
  try {
    val prefs =
      context.getSharedPreferences(HIDDEN_PREFS_KEY, Context.MODE_PRIVATE)
    val editor = prefs.edit()
    editor.putBoolean(videoId, true)
    editor.apply()
  } catch (ex: Exception) {
    Timber.e("Failed to set hidden program in the prefs (id: $videoId)")
  }
}

@TargetApi(Build.VERSION_CODES.O)
fun isProgramHiddenByUser(context: Context, videoId: String): Boolean {
  return try {
    val prefs =
      context.getSharedPreferences(HIDDEN_PREFS_KEY, Context.MODE_PRIVATE)
    prefs.getBoolean(videoId, false)
  } catch (ex: Exception) {
    Timber.e("Failed to retrieve program hidden status from prefs (id: $videoId)")
    false
  }
}
