package tv.twitch.starshot64.recommendations.legacy

import android.content.Context
import android.graphics.BitmapFactory
import android.os.Build
import timber.log.Timber
import tv.twitch.fragment.StreamFragment
import tv.twitch.starshot64.R
import tv.twitch.starshot64.recommendations.IRecommendationsStrategy
import tv.twitch.starshot64.recommendations.createStreamUri
import tv.twitch.starshot64.util.simpleHttpGet
import tv.twitch.starshot64.util.unlocalizedFormat

private const val TT_CONTENT = "legacy_launcher_recommended_streams"

class NotificationRecommendationsStrategy :
  IRecommendationsStrategy {

  companion object {
    private val logger = Timber.tag("NotifRecommStrategy")
  }

  override fun handleActivityStarted(context: Context) {
    logger.d("handleActivityStarted")
    NotificationRecommendationsWorker.schedulePeriodicWork(context)
  }

  override fun handleBootCompleted(context: Context) {
    logger.d("handleBootCompleted")
    NotificationRecommendationsWorker.schedulePeriodicWork(context)
  }

  override fun handleLocaleChanged(context: Context) {
    logger.d("handleLocaleChanged")
    NotificationRecommendationsWorker.schedulePeriodicWork(context)
  }

  override fun handleInitializePrograms(context: Context) {
    logger.d("handleInitializePrograms")
    NotificationRecommendationsWorker.schedulePeriodicWork(context)
  }

  override fun updateRecommendedStreams(context: Context, streams: Iterable<StreamFragment>) {
    logger.d("updateRecommendedStreams")

    // Create our notification channel if we're on a newer Fire OS device
    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
      createNotificationChannel(context)
    }

    // Remove previous notifications
    removeAllNotifications(context)

    // Add new programs
    streams.forEach { stream ->
      // Skip if we don't have certain necessary fields
      if (stream.broadcaster?.login == null || stream.previewImageURL == null) {
        return
      }

      val viewerCount = stream.viewersCount ?: 0
      val formattedDescription = if (stream.game?.displayName != null) {
        context.resources.getQuantityString(
          R.plurals.playing_game_for_viewers,
          viewerCount,
          unlocalizedFormat(viewerCount.toLong()),
          stream.game.displayName
        )
      } else {
        context.resources.getQuantityString(
          R.plurals.playing_for_viewers,
          viewerCount,
          unlocalizedFormat(viewerCount.toLong())
        )
      }

      // Fetch preview image, skip this rec if we can't fetch an image for it
      val previewBytes = simpleHttpGet(stream.previewImageURL) ?: return
      val previewBitmap = BitmapFactory.decodeByteArray(previewBytes, 0, previewBytes.size)

      // Fire off the notification
      scheduleNotification(
        context,
        stream.id,
        stream.broadcaster.displayName,
        formattedDescription,
        previewBitmap,
        stream.previewImageURL,
        createStreamUri(stream.broadcaster.login, TT_CONTENT)
      )
    }
  }
}
